import os
import time
import logging

import six

from .loghandler import RotatingHandler

from library import config


class SkynetFormatter(logging.Formatter):
    def __init__(self):
        logging.Formatter.__init__(self)

    def formatTime(self, record, datefmt=None):
        """
        copy paste from standard logging, except we use
        '.' to delimit milliseconds.
        :param logging.LogRecord record:
        :param str datefmt:
        :return str:
        """
        ct = self.converter(record.created)
        if datefmt:
            s = time.strftime(datefmt, ct)
        else:
            t = time.strftime("%Y-%m-%d %H:%M:%S", ct)
            s = "%s.%03d" % (t, record.msecs)
        return s

    def format(self, record):
        levelno = record.levelno
        if levelno > 5:
            level = '[%-4s]' % logging.getLevelName(levelno)
        else:
            level = '(%s)' % (str(levelno) if levelno < 0 else ' %d' % levelno)

        date = self.formatTime(record)
        message = record.getMessage()
        header = '{0} {1} [{2}]  '.format(date, level, record.name)

        if '\n' in message:
            # special case for multi-line log messages
            messageLines = message.strip().split('\n')
            line = [header + messageLines[0]]
            prepend = '%s%s' % (' ' * (len(header) - 2), ': ')
            line.extend(['%s%s' % (prepend, l) for l in messageLines[1:]])
            line = '\n'.join(line)
        else:
            line = '{header}{message}'.format(header=header, message=message)
        return line


class LogImpl(object):
    def __init__(self, filename, log_dir):
        if not os.path.exists(log_dir):
            os.makedirs(log_dir)
        base_fn = filename
        filename = os.path.join(log_dir, filename)

        cfg = config.query('skynet.services.logger')
        max_bytes = cfg.get('default', {}).get('max_size', 100 * 1024 * 1024)
        backup_count = cfg.get('default', {}).get('max_count', 14)

        max_bytes = cfg.get(base_fn, {}).get('max_size', max_bytes)
        backup_count = cfg.get(base_fn, {}).get('max_count', backup_count)

        max_bytes = self._config_kmg_convert(max_bytes)

        import sys
        sys.stderr.write('fn: %s, bytes: %s, count: %s\n' % (base_fn, max_bytes, backup_count))

        self.handler = RotatingHandler(filename, maxBytes=max_bytes, backupCount=backup_count)
        self.handler.setFormatter(SkynetFormatter())
        self.handle = self.handler.handle

    def _config_kmg_convert(self, value):
        if isinstance(value, six.string_types):
            meth = (float if '.' in value else int)
            if value.endswith('K'):
                value = 1024 * meth(value.rstrip('K'))
            elif value.endswith('M'):
                value = 1024 * 1024 * meth(value.rstrip('M'))
            elif value.endswith('G'):
                value = 1024 * 1024 * 1024 * meth(value.rstrip('G'))

        return value
