from __future__ import print_function
import argparse
import contextlib
import errno
import fcntl
import os
import socket
import struct
import sys
import threading
import time

import six

from kernel.util.console import setProcTitle
from kernel.util.errors import formatException
from kernel.util.functional import singleton
from kernel.util import logging

from .logimpl import LogImpl

from api.logger import constructLogger


LOG_FILENAME = 'logfile'


class MyFuncs(object):
    def __init__(self, log, log_dir):
        self.global_log = log
        self.log_dir = log_dir
        self.stopEvent = threading.Event()
        self.defaultLogger = LogImpl(filename=LOG_FILENAME, log_dir=log_dir)
        self.customLoggers = {}
        logging.renameLevels(4)

    def _check_filename(self, filename):
        if filename != os.path.basename(filename):
            return False
        return True

    def _get_logger(self, filename):
        if filename in self.customLoggers:
            return self.customLoggers[filename]
        self.customLoggers[filename] = LogImpl(filename=filename, log_dir=self.log_dir)
        return self.customLoggers[filename]

    def log(self, app, namespace, level, msg, filename=None):
        if namespace != 'root':
            name = "{0}:{1}".format(app, namespace)
        else:
            name = app
        record = logging.LogRecord(name, level, None, None, msg, (), None)
        if filename is None:
            self.defaultLogger.handle(record)
        elif self._check_filename(filename):
            self._get_logger(filename).handle(record)
        else:
            self.global_log.warning("Got invalid filename for log: {0}".format(filename))


def udp_logger(app, sockpath, sock, log):
    header_struct = struct.Struct('!IIIiI')
    header_size = header_struct.size
    last_sock_perm_check = time.time()

    while not app.stopEvent.isSet():
        if time.time() - last_sock_perm_check > 60:
            fstat = os.stat(sockpath)
            fmode = fstat.st_mode
            last_sock_perm_check = time.time()

            if (fmode & 0o777) != 0o777:
                log.critical('Invalid udp sock file mode: 0o%o, exiting', fmode & 0o777)
                break

        try:
            data = sock.recv(2 ** 16 - 28)
            header, body = data[:header_size], data[header_size:]
            app_len, namespace_len, filename_len, level, message_len = header_struct.unpack(header)
            app_name = six.ensure_str(body[0:app_len])
            namespace = six.ensure_str(body[app_len:app_len + namespace_len])
            filename = six.ensure_str(body[
                app_len + namespace_len:app_len + namespace_len + filename_len
            ]) if filename_len else None
            message = six.ensure_str(body[
                app_len + namespace_len + filename_len:app_len + namespace_len + filename_len + message_len
            ])
            app.log(app_name, namespace, int(level), message, filename)
        except BaseException:
            pass


def run_udp_logger_thread(path, handler, log):
    if os.path.exists(path):
        os.unlink(path)

    sock = socket.socket(socket.AF_UNIX, socket.SOCK_DGRAM)
    sock.settimeout(1)
    sock.bind(path)
    os.chmod(path, 0o777)  # set 0777 on udp socket, so others will be able to log via us

    thread = threading.Thread(target=udp_logger, args=(handler, path, sock, log))
    thread.daemon = True
    thread.start()
    return thread


def get_root():  # type: () -> str
    from srvmngr.utils import getRoot  # type: ignore
    return getRoot()


def migrate_log_names(log_dir):
    """Migrates old log file names (skynet version below 8.16.x) to the new format"""
    try:
        import py

        def _srt_key(a):
            ab = a.basename
            try:
                return int(ab.rsplit('.', 1)[1])
            except:
                return 0

        log_dir = py.path.local(log_dir)
        oldfiles = [
            log_dir.join(x)
            for x in (
                [LOG_FILENAME] +
                [LOG_FILENAME + ('.%d' % i) for i in range(1, 15)]
            )
        ]
        idx = 0

        for fn in sorted(log_dir.listdir(), key=_srt_key):
            if fn not in oldfiles:
                continue
            if fn.check(link=1):
                continue
            if not fn.check(file=1):
                continue

            if idx == 0:
                nfn = log_dir.join(LOG_FILENAME + '.' + '1970-01-01')
            else:
                nfn = log_dir.join(LOG_FILENAME + '.' + '1970-01-01' + '.' + ('%03d' % (idx, )))

            fn.rename(nfn)
            idx += 1
    except:
        pass


def parse_args(args=None):
    parser = argparse.ArgumentParser(description='Logger daemon')
    parser.add_argument('-n', '--name', default='logger', help='root logger name')
    parser.add_argument('-d', '--log-dir', default=None, help='path to directory with logs')
    parser.add_argument('-s', '--socket-path',
                        dest='socket_path',
                        default=os.getenv('SKYNET_LOGGER_SOCKET'),
                        required=os.getenv('SKYNET_LOGGER_SOCKET') is None,
                        help='ping socket path')
    parser.add_argument('-l', '--lock', required=True)

    return parser.parse_args(args)


@singleton
def cfg():
    args = parse_args()
    config = {
        'name': args.name,
        'proctitle': 'skynet.%s' % (args.name,),
        'RPCName': 'skynet.%s' % (args.name,),
        'RPCSocketPath': args.socket_path,
        'RPCLockPath': args.lock,
        'UDPSocketPath': args.socket_path + '.udp',
        'LogDir': args.log_dir if args.log_dir is not None else os.path.join(get_root(), 'var', 'log')
    }
    return config


@contextlib.contextmanager
def lock(path):
    try:
        try:
            fp = open(path, 'wb')
        except IOError as ex:
            if ex.errno == errno.ENOENT:
                os.makedirs(os.path.dirname(path))
                fp = open(path, 'wb')
            else:
                raise

        try:
            fcntl.flock(fp, fcntl.LOCK_EX | fcntl.LOCK_NB)
        except IOError:
            raise Exception('Lock %s is held by somebody else' % (path, ))

        yield
    finally:
        pass


def ping_responder(sock):
    sock.listen(8)

    while True:
        conn, peer = sock.accept()
        buff = []

        while True:
            ch = conn.recv(1)
            if ch == '\n' or not ch:
                break
            buff.append(ch)

        data = ''.join(buff)

        if data == 'PING':
            conn.sendall('OKAY\n')
            conn.shutdown(socket.SHUT_RDWR)


def main():
    config = cfg()
    setProcTitle(config['proctitle'])

    log = constructLogger(app=config['name'])
    migrate_log_names(log_dir=config['LogDir'])

    handler = MyFuncs(log, log_dir=config['LogDir'])

    # Replace global logger with direct one
    from .iface import Logger
    Logger.log = handler.log

    log.info('initializing...')

    result = 127  # unpredictable error

    try:
        with lock(config['RPCLockPath']):
            lock_socket = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
            try:
                lock_socket.bind(config['RPCSocketPath'])
            except socket.error as ex:
                if ex.errno == errno.ENOENT:
                    os.makedirs(os.path.dirname(config['RPCSocketPath']))
                elif ex.errno == errno.EADDRINUSE:
                    os.unlink(config['RPCSocketPath'])
                else:
                    raise

                lock_socket.bind(config['RPCSocketPath'])

            ping_responder_thr = threading.Thread(target=ping_responder, args=(lock_socket, ))
            ping_responder_thr.daemon = True
            ping_responder_thr.start()

            udp_logger_thread = run_udp_logger_thread(config['UDPSocketPath'], handler, log=log)
            log.info('daemon started')

            try:
                while not handler.stopEvent.isSet():
                    handler.stopEvent.wait(timeout=1)
                    if not udp_logger_thread.is_alive():
                        break
            except KeyboardInterrupt:
                print('interrupted, exiting', file=sys.stderr)
                log.info('got SIGINT, exiting')
                handler.stopEvent.set()

            if not handler.stopEvent.isSet():
                log.critical('exiting -- udp or rpc thread died')
                handler.stopEvent.set()

            udp_logger_thread.join(timeout=5)
            if udp_logger_thread.is_alive():
                log.critical('udp logger thread is still alive (after 5 secs) -- critical shutdown')
                result = 1
            else:
                result = 0

    except KeyboardInterrupt:
        print('interrupted', file=sys.stderr)
        result = 1

    except Exception:
        try:
            log.critical('exiting on error: {0}'.format(formatException()))
            print(formatException(), file=sys.stderr)
        except BaseException:
            pass
        result = 1

    try:
        try:
            os.unlink(config['RPCSocketPath'])
        except:
            pass

        try:
            os.unlink(config['RPCLockPath'])
        except:
            pass
    finally:
        os._exit(result)


if __name__ == '__main__':
    raise SystemExit(main())
