package main

import (
	"fmt"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/maxwell/go/pkg/maxwell/client"
	"a.yandex-team.ru/infra/maxwell/go/pkg/yamlutils"
	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

func registerJobs(root *cobra.Command, config *maxctlConfig) {
	showSpec := false
	showMeta := false
	showStatus := false

	cmd := &cobra.Command{
		Use:   "jobs",
		Short: "Jobs managing",
		Args:  cobra.NoArgs,
	}

	cmdPut := &cobra.Command{
		Use:   "put <path-to-spec.yaml>",
		Short: "Start/update job",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			maxwellToken := getMaxwellTokenFromEnv()
			var m client.Maxwell
			var err error
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			if config.url == "" {
				m, err = client.CreateMaxwell(URLProduction, maxwellToken, l)
			} else {
				m, err = client.CreateMaxwell(config.url, maxwellToken, l)
			}
			if err != nil {
				er(err)
			}
			spec := &pb.Job_Spec{}
			specPath := args[0]
			err = yamlutils.ReadFromFile(specPath, spec)
			if err != nil {
				er(err)
			}
			resp, err := m.PutJob(spec)
			if err != nil {
				er(err)
			}
			fmt.Println(resp)
		},
	}

	cmdRestore := &cobra.Command{
		Use:   "restore <id>",
		Short: "Restore job from cockroach",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			maxwellToken := getMaxwellTokenFromEnv()
			var m client.Maxwell
			var err error
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			if config.url == "" {
				m, err = client.CreateMaxwell(URLProduction, maxwellToken, l)
			} else {
				m, err = client.CreateMaxwell(config.url, maxwellToken, l)
			}
			if err != nil {
				er(err)
			}
			id := args[0]
			resp, err := m.RestoreJob(id)
			if err != nil {
				er(err)
			}
			fmt.Println(string(resp))
		},
	}

	cmdStop := &cobra.Command{
		Use:   "stop <jobID>",
		Short: "Stop job",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			maxwellToken := getMaxwellTokenFromEnv()
			var m client.Maxwell
			var err error
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			if config.url == "" {
				m, err = client.CreateMaxwell(URLProduction, maxwellToken, l)
			} else {
				m, err = client.CreateMaxwell(config.url, maxwellToken, l)
			}
			if err != nil {
				er(err)
			}
			resp, err := m.StopJob(args[0])
			if err != nil {
				er(err)
			}
			fmt.Println(resp)
		},
	}

	cmdDelete := &cobra.Command{
		Use:   "delete <name>",
		Short: "Delete job",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			maxwellToken := getMaxwellTokenFromEnv()
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			url := URLProduction
			if config.url == "" {
				url = config.url
			}
			m, err := client.CreateMaxwell(url, maxwellToken, l)
			if err != nil {
				er(err)
			}
			if err := m.DeleteJob(args[0]); err != nil {
				er(err)
			}
			fmt.Println("Job was totally removed")
		},
	}

	cmdList := &cobra.Command{
		Use:   "list",
		Short: "List all jobs",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			maxwellToken := getMaxwellTokenFromEnv()
			var m client.Maxwell
			var err error
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			if config.url == "" {
				m, err = client.CreateMaxwell(URLProduction, maxwellToken, l)
			} else {
				m, err = client.CreateMaxwell(config.url, maxwellToken, l)
			}
			if err != nil {
				er(err)
			}
			jobs, err := m.GetJobs()
			if err != nil {
				er(err)
			}
			spec := ""
			meta := ""
			status := ""
			for _, j := range jobs.Jobs {
				if showSpec {
					specBytes, err := yamlutils.ProtoToYaml(j.Spec)
					if err != nil {
						er(err)
					}
					spec = string(specBytes)
				}
				if showMeta {
					metaBytes, err := yamlutils.ProtoToYaml(j.Meta)
					if err != nil {
						er(err)
					}
					meta = string(metaBytes)
				}
				if showStatus {
					statusBytes, err := yamlutils.ProtoToYaml(j.Status)
					if err != nil {
						er(err)
					}
					status = string(statusBytes)
				}
				fmt.Println(colorize(j.Spec.Name, redTextColor, boldText))
				if showSpec {
					fmt.Println(colorize("spec", yellowTextColor, boldText))
					fmt.Println(spec)
				}
				if showMeta {
					fmt.Println(colorize("meta", yellowTextColor, boldText))
					fmt.Println(meta)
				}
				if showStatus {
					fmt.Println(colorize("status", yellowTextColor, boldText))
					fmt.Println(status)
				}
			}
		},
	}

	cmd.AddCommand(cmdPut)
	cmd.AddCommand(cmdStop)
	cmd.AddCommand(cmdDelete)
	cmd.AddCommand(cmdList)
	cmd.AddCommand(cmdRestore)

	cmdList.Flags().BoolVar(&showSpec, "spec", false, "Show spec")
	cmdList.Flags().BoolVar(&showMeta, "meta", false, "Show meta")
	cmdList.Flags().BoolVar(&showStatus, "status", false, "Show status")

	root.AddCommand(cmd)
}

const (
	redTextColor    = "31"
	yellowTextColor = "33"
	boldText        = "1"
)

func colorize(text string, modification ...string) string {
	return fmt.Sprintf("\033[%sm%s\033[0m", strings.Join(modification, ";"), text)
}
