package main

import (
	"archive/tar"
	"bytes"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"path"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/maxwell/go/pkg/maxwell/client"
	"a.yandex-team.ru/infra/maxwell/go/pkg/yamlutils"
	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

func registerSpecs(root *cobra.Command, config *maxctlConfig) {
	dumpPath := ""

	cmd := &cobra.Command{
		Use:   "specs",
		Short: "Specs managing",
		Args:  cobra.NoArgs,
	}

	cmdDump := &cobra.Command{
		Use:   "dump",
		Short: "Dump all jobs specs",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			maxwellToken := getMaxwellTokenFromEnv()
			var m client.Maxwell
			var err error
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			if config.url == "" {
				m, err = client.CreateMaxwell(URLProduction, maxwellToken, l)
			} else {
				m, err = client.CreateMaxwell(config.url, maxwellToken, l)
			}
			if err != nil {
				er(err)
			}
			jobs, err := m.GetJobs()
			if err != nil {
				er(err)
			}
			tarFile, err := os.Create(dumpPath)
			if err != nil {
				er(err)
			}
			defer tarFile.Close()
			tw := tar.NewWriter(tarFile)
			defer tw.Close()
			for _, j := range jobs.Jobs {
				jobYAML, err := yamlutils.ProtoToYaml(j.Spec)
				if err != nil {
					er(err)
					return
				}
				hdr := &tar.Header{
					Name: j.Spec.Name + ".yaml",
					Mode: 0600,
					Size: int64(len(jobYAML)),
				}
				if err := tw.WriteHeader(hdr); err != nil {
					er(err)
					return
				}
				if _, err := tw.Write(jobYAML); err != nil {
					er(err)
					return
				}
			}
		},
	}

	cmdDir := &cobra.Command{
		Use:   "dir <dir>",
		Short: "Start all specs from <dir>",
		Args:  cobra.ExactArgs(1),
		Run: func(cmd *cobra.Command, args []string) {
			maxwellToken := getMaxwellTokenFromEnv()
			var m client.Maxwell
			var err error
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			if config.url == "" {
				m, err = client.CreateMaxwell(URLProduction, maxwellToken, l)
			} else {
				m, err = client.CreateMaxwell(config.url, maxwellToken, l)
			}
			if err != nil {
				er(err)
			}
			entries, err := ioutil.ReadDir(args[0])
			if err != nil {
				er(err)
			}
			for _, entry := range entries {
				f, err := os.Open(path.Join(args[0], entry.Name()))
				if err != nil {
					fmt.Printf("failed to open '%s': %s\n", entry.Name(), err)
					continue
				}
				content, err := ioutil.ReadAll(f)
				if err != nil {
					fmt.Printf("failed to read '%s': %s\n", entry.Name(), err)
					continue
				}
				spec := &pb.Job_Spec{}
				err = yamlutils.YAMLToProto(content, spec)
				if err != nil {
					fmt.Printf("Failed to unmarshal: %s\n", err.Error())
					continue
				}
				resp, err := m.PutJob(spec)
				if err != nil {
					fmt.Printf("Failed to start %s: %s\n", spec.Name, err.Error())
					continue
				}
				fmt.Printf("Starting job '%s': %s\n", spec.Name, resp)
			}
		},
	}

	cmdRestore := &cobra.Command{
		Use:   "restore",
		Short: "Restore all jobs specs from dump",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			maxwellToken := getMaxwellTokenFromEnv()
			var m client.Maxwell
			var err error
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			if config.url == "" {
				m, err = client.CreateMaxwell(URLProduction, maxwellToken, l)
			} else {
				m, err = client.CreateMaxwell(config.url, maxwellToken, l)
			}
			if err != nil {
				er(err)
			}
			file, err := os.Open(dumpPath)
			if err != nil {
				er(err)
			}
			defer file.Close()
			var fileReader io.ReadCloser = file
			tr := tar.NewReader(fileReader)
			for {
				header, err := tr.Next()
				if err == io.EOF {
					break
				}
				if err != nil {
					os.Exit(1)
				}
				switch header.Typeflag {
				case tar.TypeDir:
					continue
				case tar.TypeReg:
					buf := &bytes.Buffer{}
					_, err = io.Copy(buf, tr)
					if err != nil {
						er(err)
					}
					spec := &pb.Job_Spec{}
					err := yamlutils.YAMLToProto(buf.Bytes(), spec)
					if err != nil {
						fmt.Printf("Failed to unmarshal: %s", err.Error())
					}
					resp, err := m.PutJob(spec)
					if err != nil {
						fmt.Printf("Failed to start %s: %s", spec.Name, err.Error())
					}
					fmt.Printf("Starting job '%s': %s\n", spec.Name, resp)
				}
			}
		},
	}

	cmd.AddCommand(cmdDump)
	cmd.AddCommand(cmdRestore)
	cmd.AddCommand(cmdDir)

	cmdDump.Flags().StringVar(&dumpPath, "dump-path", "jobs_specs_dump.tar", "Dump specs")
	cmdRestore.Flags().StringVar(&dumpPath, "dump-path", "jobs_specs_dump.tar", "File to restore specs from")

	root.AddCommand(cmd)
}
