package api

import (
	"context"
	"fmt"
	"net/http"
	"os"

	"github.com/go-chi/chi/v5"
	"google.golang.org/grpc"
	"google.golang.org/grpc/grpclog"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/maxwell/go/internal/httprpc"
	"a.yandex-team.ru/infra/maxwell/go/internal/job"
	"a.yandex-team.ru/infra/maxwell/go/internal/pbutil"
	"a.yandex-team.ru/infra/maxwell/go/internal/validators"
	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox"
)

const (
	runningState = "running"
	stoppedState = "stopped"
)

func NewJobmanAPI(m *job.Manager, bb blackbox.Client, insecure bool) *JobmanAPI {
	s := grpc.NewServer()
	glog := grpclog.NewLoggerV2(os.Stdout, os.Stdout, os.Stdout)
	grpclog.SetLoggerV2(glog)
	l, _ := zap.New(zap.ConsoleConfig(log.InfoLevel))
	serv := &JobmanAPI{
		m:        m,
		l:        l,
		s:        s,
		bb:       bb,
		insecure: insecure,
	}
	return serv
}

type JobmanAPI struct {
	m        *job.Manager
	l        log.Logger
	s        *grpc.Server
	bb       blackbox.Client
	insecure bool
}

func (s *JobmanAPI) ListJobs(_ context.Context, _ *pb.ListJobsRequest) (*pb.ListJobsResponse, error) {
	jobs, err := s.m.List()
	if err != nil {
		return nil, err
	}
	names := make([]string, len(jobs))
	for i := 0; i < len(jobs); i++ {
		names[i] = jobs[i].Spec.Name
	}
	return &pb.ListJobsResponse{Jobs: names}, nil
}

func (s *JobmanAPI) GetJob(_ context.Context, req *pb.GetJobRequest) (*pb.GetJobResponse, error) {
	j, err := s.m.Get(req.Name)
	if err != nil {
		return nil, err
	}
	if j == nil {
		return nil, fmt.Errorf("'%s' not found", req.Name)
	}
	return &pb.GetJobResponse{Job: fullJobToJobInfo(j)}, nil
}

func fullJobToJobInfo(full *pb.FullJob) *pb.JobInfo {
	limit := 200 // max hosts to send in ui
	remaining := 0
	for _, group := range full.Groups {
		remaining += len(group.Hosts)
	}
	head := make([]*pb.JobInfo_Queue, 0)
	currentCount := 0
	for _, group := range full.Groups {
		if currentCount+len(group.Hosts) > limit {
			head = append(head, &pb.JobInfo_Queue{
				Group: group.Group,
				Hosts: group.Hosts[:limit-currentCount],
			})
			break
		} else {
			head = append(head, &pb.JobInfo_Queue{
				Group: group.Group,
				Hosts: group.Hosts,
			})
			currentCount += len(group.Hosts)
		}
	}
	return &pb.JobInfo{
		Job:        full.Job,
		WorkingSet: full.WorkingSet,
		Head:       head,
		Remaining:  int32(remaining),
		Processed:  int32(len(full.State.Processed)),
	}
}

func (s *JobmanAPI) GetJobsSummary(_ context.Context, _ *pb.GetJobsSummaryRequest) (*pb.GetJobsSummaryResponse, error) {
	summaries := make([]*pb.Job, 0)
	jobs, err := s.m.List()
	if err != nil {
		return nil, err
	}
	summaries = append(summaries, jobs...)
	return &pb.GetJobsSummaryResponse{Jobs: summaries}, nil
}

func (s *JobmanAPI) PutJob(ctx context.Context, reqCtx *pb.RequestCtx, req *pb.PutJobRequest) (*pb.PutJobResponse, error) {
	if err := validators.SpecIsValid(req.Spec); err != nil {
		return nil, fmt.Errorf("spec is not valid: %s", err)
	}
	username, err := s.getUsername(ctx, reqCtx)
	if err != nil {
		return nil, err
	}
	if err := s.m.PutJob(req.Spec, username); err != nil {
		return nil, err
	}
	return &pb.PutJobResponse{}, nil
}

func (s *JobmanAPI) StopJob(_ context.Context, req *pb.StopJobRequest) (*pb.StopJobResponse, error) {
	j, err := s.m.Summary(req.Name)
	if err != nil {
		return nil, err
	}
	pbutil.SetFalse(j.Status.Running, "stopped")
	j.Status.RememberedState = stoppedState
	if err := s.m.Update(j); err != nil {
		return nil, err
	}
	return nil, err
}

func (s *JobmanAPI) ResumeJob(ctx context.Context, reqCtx *pb.RequestCtx, req *pb.ResumeJobRequest) (*pb.ResumeJobResponse, error) {
	username, err := s.getUsername(ctx, reqCtx)
	if err != nil {
		return nil, err
	}
	j, err := s.m.Summary(req.Name)
	if err != nil {
		return nil, err
	}
	pbutil.SetTrue(j.Status.Running, "running")
	j.Status.RememberedState = runningState
	j.Meta.Author = username
	if err := s.m.Update(j); err != nil {
		return nil, err
	}
	return nil, err
}

func (s *JobmanAPI) DeleteJob(_ context.Context, req *pb.DeleteJobRequest) (*pb.DeleteJobResponse, error) {
	err := s.m.Delete(req.Name)
	if err != nil {
		return nil, err
	}
	return nil, err
}

func (s *JobmanAPI) StopAll(_ context.Context, _ *pb.StopAllRequest) (*pb.StopAllResponse, error) {
	jobs, err := s.m.List()
	if err != nil {
		return nil, err
	}
	stopped := make([]string, 0)
	for _, j := range jobs {
		if j.Status.Running.Status != "False" {
			pbutil.SetFalse(j.Status.Running, "Stopped by 'Stop all' button")
			if err := s.m.Update(j); err != nil {
				return nil, err
			}
			stopped = append(stopped, j.Spec.Name)
		}
	}
	return &pb.StopAllResponse{Stopped: stopped}, nil
}

func (s *JobmanAPI) ResumeAll(_ context.Context, _ *pb.ResumeAllRequest) (*pb.ResumeAllResponse, error) {
	jobs, err := s.m.List()
	if err != nil {
		return nil, err
	}
	resumed := make([]string, 0)
	for _, j := range jobs {
		if j.Status.Running.Status == "False" && j.Status.RememberedState == runningState {
			pbutil.SetTrue(j.Status.Running, "Running by 'Resume all' button")
			if err := s.m.Update(j); err != nil {
				return nil, err
			}
			resumed = append(resumed, j.Spec.Name)
		}
	}
	return &pb.ResumeAllResponse{Resumed: resumed}, nil
}

// MuteTask for job. Muted tasks will not counted in processing tasks (tasks in working set)
func (s *JobmanAPI) MuteTask(ctx context.Context, reqCtx *pb.RequestCtx, r *pb.MuteTaskRequest) (*pb.MuteTaskResponse, error) {
	username, err := s.getUsername(ctx, reqCtx)
	if err != nil {
		return nil, err
	}
	err = s.m.MuteTask(r.Job, r.Hostname, username)
	if err != nil {
		return nil, err
	}
	return &pb.MuteTaskResponse{}, nil
}

func (s *JobmanAPI) Master(_ context.Context, _ *pb.RequestCtx, _ *pb.MasterRequest) (*pb.MasterResponse, error) {
	master, err := s.m.Master()
	if err != nil {
		return nil, err
	}
	return &pb.MasterResponse{Master: master}, nil
}

func (s *JobmanAPI) getUsername(ctx context.Context, reqCtx *pb.RequestCtx) (string, error) {
	user, err := bbGetUsername(ctx, reqCtx, s.bb)
	if !s.insecure && err != nil {
		s.l.Error(err.Error())
		return "", err
	}
	username := "insecure"
	if !s.insecure {
		username = user.Login
	}
	return username, nil
}

func (s *JobmanAPI) Register(mux *chi.Mux) {
	httprpc.New("GET", "/api/jobs").
		CorsAllowAll().
		WithMultiTypeReader(&pb.ListJobsRequest{}).
		WithHandler(func(ctx context.Context, _ *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.ListJobs(ctx, req.(*pb.ListJobsRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("GET", "/api/jobs/{name}").
		CorsAllowAll().
		WithRequestReader(func(r *http.Request) (proto.Message, *pb.RequestCtx, error) {
			return &pb.GetJobRequest{Name: chi.URLParam(r, "name")}, nil, nil
		}).
		WithHandler(func(ctx context.Context, _ *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.GetJob(ctx, req.(*pb.GetJobRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("GET", "/api/jobs-summary").
		CorsAllowAll().
		WithMultiTypeReader(&pb.GetJobsSummaryRequest{}).
		WithHandler(func(ctx context.Context, _ *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.GetJobsSummary(ctx, req.(*pb.GetJobsSummaryRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("POST", "/api/jobs").
		CorsAllowAll().
		WithMultiTypeReader(&pb.PutJobRequest{}).
		WithHandler(func(ctx context.Context, reqCtx *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.PutJob(ctx, reqCtx, req.(*pb.PutJobRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("POST", "/api/jobs/{name}/resume").
		CorsAllowAll().
		WithRequestReader(func(r *http.Request) (proto.Message, *pb.RequestCtx, error) {
			return &pb.ResumeJobRequest{Name: chi.URLParam(r, "name")}, httprpc.RequestCtxFromRequest(r), nil
		}).
		WithHandler(func(ctx context.Context, reqCtx *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.ResumeJob(ctx, reqCtx, req.(*pb.ResumeJobRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("DELETE", "/api/jobs/{name}").
		CorsAllowAll().
		WithRequestReader(func(r *http.Request) (proto.Message, *pb.RequestCtx, error) {
			return &pb.StopJobRequest{Name: chi.URLParam(r, "name")}, nil, nil
		}).
		WithHandler(func(ctx context.Context, reqCtx *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.StopJob(ctx, req.(*pb.StopJobRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("DELETE", "/api/jobs/{name}/delete").
		CorsAllowAll().
		WithRequestReader(func(r *http.Request) (proto.Message, *pb.RequestCtx, error) {
			return &pb.DeleteJobRequest{Name: chi.URLParam(r, "name")}, nil, nil
		}).
		WithHandler(func(ctx context.Context, reqCtx *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.DeleteJob(ctx, req.(*pb.DeleteJobRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("GET", "/api/jobs/stopall").
		CorsAllowAll().
		WithMultiTypeReader(&pb.StopAllRequest{}).
		WithHandler(func(ctx context.Context, reqCtx *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.StopAll(ctx, req.(*pb.StopAllRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("GET", "/api/jobs/resumeall").
		CorsAllowAll().
		WithMultiTypeReader(&pb.ResumeAllRequest{}).
		WithHandler(func(ctx context.Context, reqCtx *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.ResumeAll(ctx, req.(*pb.ResumeAllRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("GET", "/api/master").
		CorsAllowAll().
		WithMultiTypeReader(&pb.MasterRequest{}).
		WithHandler(func(ctx context.Context, reqCtx *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.Master(ctx, reqCtx, req.(*pb.MasterRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
	httprpc.New("POST", "/api/MuteTask").
		CorsAllowAll().
		WithMultiTypeReader(&pb.MuteTaskRequest{}).
		WithHandler(func(ctx context.Context, reqCtx *pb.RequestCtx, req proto.Message) (proto.Message, error) {
			return s.MuteTask(ctx, reqCtx, req.(*pb.MuteTaskRequest))
		}).
		WithWriterByAccepted().
		Mount(mux)
}
