package api

import (
	"context"
	"fmt"
	"net"
	"net/http"

	"github.com/golang/protobuf/jsonpb"

	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/yandex/blackbox"
)

var mrshlr = jsonpb.Marshaler{
	OrigName:     false,
	EmitDefaults: true,
	Indent:       "  ",
}

func bbCheckSession(ctx context.Context, bb blackbox.Client, sessionID, remoteAddr string) (*blackbox.SessionIDResponse, error) {
	host, _, _ := net.SplitHostPort(remoteAddr)
	bbResponse, err := bb.SessionID(ctx, blackbox.SessionIDRequest{
		SessionID: sessionID,
		UserIP:    host,
		Host:      "maxwell.in.yandex-team.ru",
	})
	if err != nil {
		return nil, err
	}
	return bbResponse, nil
}

func bbCheckOauth(ctx context.Context, bb blackbox.Client, token, remoteAddr string) (*blackbox.OAuthResponse, error) {
	if token == "" {
		return nil, fmt.Errorf("empty oauth token")
	}
	host, _, _ := net.SplitHostPort(remoteAddr)
	bbResponse, err := bb.OAuth(ctx, blackbox.OAuthRequest{
		OAuthToken: token,
		UserIP:     host,
	})
	if err != nil {
		return nil, err
	}
	return bbResponse, nil
}

func bbGetUsername(ctx context.Context, reqCtx *pb.RequestCtx, bb blackbox.Client) (*blackbox.User, error) {
	sessionResp, err1 := bbCheckSession(ctx, bb, reqCtx.SessionId, reqCtx.RemoteAddr)
	if err1 != nil {
		oauthResp, err2 := bbCheckOauth(ctx, bb, reqCtx.OauthToken, reqCtx.RemoteAddr)
		if err2 != nil {
			return nil, fmt.Errorf("failed to get username: %s:%s", err1, err2)
		}
		return &oauthResp.User, nil
	}
	return &sessionResp.User, nil
}

func internalError(err error, w http.ResponseWriter) {
	w.WriteHeader(500)
	_, _ = w.Write([]byte(err.Error()))
}

func er(err error, w http.ResponseWriter, statusCode int) {
	w.WriteHeader(statusCode)
	_, _ = w.Write([]byte(err.Error()))
}
