package api

import (
	"io/ioutil"
	"net"
	"net/http"
	"net/url"
	"time"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/infra/maxwell/ui"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/blackbox"
)

type LoginHandler struct {
	l            *zap.Logger
	bb           blackbox.Client
	clientID     string
	clientSecret string
	insecure     bool
}

func NewLoginHandler(bb blackbox.Client, clientID, clientSecret string, insecure bool) *LoginHandler {
	l, _ := zap.New(zap.ConsoleConfig(log.InfoLevel))
	return &LoginHandler{l, bb, clientID, clientSecret, insecure}
}

func (h *LoginHandler) LoginHandler(w http.ResponseWriter, r *http.Request) {
	if h.insecure {
		cookie := http.Cookie{Name: "yandex_login", Value: "insecure", Expires: time.Now().Add(time.Hour)}
		http.SetCookie(w, &cookie)
		http.Redirect(w, r, "/", http.StatusFound)
		return
	}
	sessID, err := r.Cookie("Session_id")
	if err != nil {
		http.Redirect(w, r, "https://passport.yandex-team.ru/auth?origin=maxwell&retpath=https://maxwell.in.yandex-team.ru/login", http.StatusFound)
		return
	}
	host, _, _ := net.SplitHostPort(r.RemoteAddr)
	_, err = h.bb.SessionID(r.Context(), blackbox.SessionIDRequest{
		SessionID: sessID.Value,
		UserIP:    host,
		Host:      "maxwell.in.yandex-team.ru",
	})
	if err != nil {
		if blackbox.IsUnauthorized(err) {
			http.Redirect(w, r, "https://passport.yandex-team.ru/auth?origin=maxwell&retpath=https://maxwell.in.yandex-team.ru/login", http.StatusFound)
		}
		err = xerrors.Errorf("failed to check bbResponse session: %w", err)
		internalError(err, w)
	}
	ui.SpaHandler(w, r)
}

func (h *LoginHandler) OAuthHandler(w http.ResponseWriter, r *http.Request) {
	code := r.Header.Get("OAuthCode")
	resp, err := http.PostForm("https://oauth.yandex-team.ru/token", url.Values{"grant_type": {"authorization_code"}, "code": {code}, "client_id": {h.clientID}, "client_secret": {h.clientSecret}})
	if err != nil {
		err = xerrors.Errorf("invalid token: %w", err)
		internalError(err, w)
	}
	respBytes, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		err = xerrors.Errorf("invalid token: %w", err)
		internalError(err, w)
	}
	w.Header().Set("Content-Type", headerValueJSON)
	w.Header().Set("Content-Length", resp.Header.Get("Content-Length"))
	if _, err := w.Write(respBytes); err != nil {
		h.l.Errorf("Failed to write response: %e", err)
	}
}

func (h *LoginHandler) Register(mux *chi.Mux) {
	mux.Get("/login", h.LoginHandler)
	mux.Post("/oauthCode", h.OAuthHandler)
}
