package job

import (
	"fmt"
	"math"
	"strings"
	"time"

	"github.com/golang/protobuf/ptypes"

	"a.yandex-team.ru/infra/maxwell/go/internal/pbutil"
	"a.yandex-team.ru/infra/maxwell/go/internal/pool"
	"a.yandex-team.ru/infra/maxwell/go/internal/pool/order"
	"a.yandex-team.ru/infra/maxwell/go/internal/storages"
	"a.yandex-team.ru/infra/maxwell/go/internal/tasks"
	"a.yandex-team.ru/infra/maxwell/go/internal/validators"
	"a.yandex-team.ru/infra/maxwell/go/internal/workingset"
	"a.yandex-team.ru/infra/maxwell/go/pkg/walle"
	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
)

func NewController(j *pb.Job, jobStorage storages.Job, jobsStorage storages.Jobs, hostsStorage storages.Hosts,
	nannyStorage storages.NannyServices, ypStorage storages.YpNodes, dryRun bool) *Controller {
	spec := j.Spec
	return &Controller{
		j:           j,
		dryRun:      dryRun,
		ws:          workingset.New(spec.Name, jobStorage, hostsStorage, j.Spec),
		pool:        pool.New(spec, jobStorage, nannyStorage, ypStorage),
		jobsStorage: jobsStorage,
		jobStorage:  jobStorage,
		ypStorage:   ypStorage,
	}
}

type Controller struct {
	j      *pb.Job
	dryRun bool

	ws          *workingset.WorkingSet
	pool        *pool.Pool
	jobsStorage storages.Jobs
	jobStorage  storages.Job
	ypStorage   storages.YpNodes
}

/*
RefreshWorkingSet:
* Update job's conditions (running, sleep, paused)
* Update hosts in working set
* Remove finished hosts from working set
* Prepare tasks on hosts from queue
* Start prepared tasks
*/
func (c *Controller) RefreshWorkingSet(w walle.IClient, l log.Logger) error {
	l = l.WithName(c.j.Spec.Name)
	l.Infof("Restoring working set...")
	if err := c.ws.Restore(); err != nil {
		return err
	}
	if err := c.UpdateWorkingSet(w, l); err != nil {
		return err
	}
	l.Infof("Filtering working set from finished tasks...")
	if err := c.RemoveFinished(l); err != nil {
		return err
	}
	// Update job status conditions
	running, err := c.isRunning(l)
	if err != nil {
		return err
	}
	// Skip run on paused, not running, sleeping conditions
	if !running {
		status := c.j.Status
		l.Infof("Skipping update running=%s sleep=%s", status.Running.Status, status.Sleep.Status)
		return nil
	}
	l.Infof("Preparing tasks...")
	if err := c.prepareTasks(w, l); err != nil {
		return err
	}
	l.Infof("Starting tasks...")
	if err := c.startTasks(w, l); err != nil {
		return err
	}
	l.Infof("Canceling tasks for enforce...")
	if err := c.cancelTasks(w, l); err != nil {
		return err
	}
	l.Infof("Enforcing tasks...")
	if err := c.enforceTasks(w, l); err != nil {
		return err
	}
	return nil
}

// UpdateWorkingSet updating hosts statuses from wall-e
func (c *Controller) UpdateWorkingSet(w walle.IClient, l log.Logger) error {
	l.Infof("Updating working set...")
	if err := c.ws.Update(w, l); err != nil {
		return err
	}
	l.Infof("Dumping working set...")
	if err := c.ws.Dump(); err != nil {
		return err
	}
	return nil
}

// Removes finished tasks from working set
func (c *Controller) RemoveFinished(l log.Logger) error {
	filtered, err := c.ws.Filter(func(task *pb.Task) bool { return task.State != pb.Task_FINISHED })
	if err != nil {
		return err
	}
	if len(filtered) == 0 {
		return nil
	}
	hostnames := make([]string, len(filtered))
	for i, task := range filtered {
		hostnames[i] = task.Hostname
	}
	l.Infof("[%s] to be removed from working set, because state=FINISHED", strings.Join(hostnames, ", "))
	l.Infof("Dumping working set...")
	if err := c.ws.Dump(); err != nil {
		return err
	}
	if c.j.Spec.PostCondition.Delay != nil {
		if d, _ := ptypes.Duration(c.j.Spec.PostCondition.Delay.Duration); d != 0 {
			pbutil.SetTrue(c.j.Status.Sleep, fmt.Sprintf("Sleep %s since: %s", d, time.Now()))
			if err := c.jobsStorage.Put(c.j); err != nil {
				return err
			}
		}
	}
	if err := c.jobStorage.AppendProcessed(hostnames); err != nil {
		return err
	}
	return nil
}

/*
* Refetch all by spec.source
* Filter all by spec.filters
* Order
* Groups
* Save
 */
func (c *Controller) UpdatePool(w walle.IClient, l log.Logger) error {
	startTime := time.Now()
	l.Infof("Starting ws.Restore()")
	if err := c.ws.Restore(); err != nil {
		return err
	}
	records := c.ws.Records()
	l.Infof("Finished ws.Restore() in %fs", time.Since(startTime).Seconds())
	filterHostNames := make([]string, 0, len(records))
	for i := 0; i < len(records); i++ {
		filterHostNames = append(filterHostNames, records[i].Task.Hostname)
	}
	if c.j.Spec.Mode == "script" {
		processed, err := c.jobStorage.Processed()
		if err != nil {
			return err
		}
		for i := 0; i < len(processed); i++ {
			filterHostNames = append(filterHostNames, processed[i])
		}
	}
	startTime = time.Now()
	l.Infof("Starting pool.Update()")
	if err := c.pool.Update(w, l, filterHostNames); err != nil {
		return err
	}
	l.Infof("Finished pool.Update() in %fs", time.Since(startTime).Seconds())
	startTime = time.Now()
	l.Infof("Starting pool.Dump()")
	if err := c.pool.Dump(l); err != nil {
		return err
	}
	l.Infof("Finished pool.Dump() in %fs", time.Since(startTime).Seconds())
	return nil
}

// MuteTask for job. Muted tasks will not counted in processing tasks (tasks in working set)
func (c *Controller) MuteTask(hostname, author string) error {
	if err := c.ws.Restore(); err != nil {
		return err
	}
	// Only one muted task allowed
	if mutedCount := len(c.ws.Records().Muted()); mutedCount > 0 {
		return fmt.Errorf("job allready has %d muted tasks: only 1 allowed", mutedCount)
	}
	r := c.ws.Records().Record(hostname)
	if r == nil {
		return fmt.Errorf("task on %s not found", hostname)
	}
	// Now we allow mute task only for repairing hosts in ITDC
	if !strings.HasPrefix(strings.ToLower(r.Host.Ticket), "itdc") {
		return fmt.Errorf("task on %s do not have ITDC ticket", hostname)
	}
	c.ws.MuteTask(hostname, author)
	return c.ws.Dump()
}

// Check if job running (sleep = False, running = True, paused = False)
func (c *Controller) isRunning(l log.Logger) (bool, error) {
	l.Infof("Updating sleep condition...")
	if err := c.updateSleepCond(l); err != nil {
		return false, err
	}
	l.Infof("Persisting job with updated statuses...")
	if err := c.jobsStorage.Put(c.j); err != nil {
		return false, err
	}
	status := c.j.Status
	return pbutil.True(status.Running) && pbutil.False(status.Sleep), nil
}

// Check if post_condition.delay expired
func (c *Controller) updateSleepCond(l log.Logger) error {
	status := c.j.Status
	if pbutil.True(status.Sleep) {
		sleepStart, err := ptypes.Timestamp(status.Sleep.TransitionTime)
		if err != nil {
			return err
		}
		duration, err := ptypes.Duration(c.j.Spec.PostCondition.Delay.Duration)
		if err != nil {
			return err
		}
		if sleepStart.Add(duration).Before(time.Now()) {
			pbutil.SetFalse(status.Sleep, "")
		}
	}
	return nil
}

/*
* Getting new hosts from queue
* Preparing tasks (for auto tasks choose needed action)
* Persist in storage
 */
func (c *Controller) prepareTasks(w walle.IClient, l log.Logger) error {
	// window - running_tasks 'without muted tasks'
	windowFree := int(c.j.Spec.Window) - len(c.ws.Records().Running().SkipMuted())
	// (window + overflow_window) - (running_tasks + overflow_tasks) 'without muted tasks'
	overflowFree := int(c.j.Spec.Window+c.j.Spec.WindowOverflow) -
		len(c.ws.Records().
			Filter([]pb.Task_State{pb.Task_RUNNING, pb.Task_WAITING}).
			SkipMuted())
	prepareCount := int(math.Min(float64(windowFree), float64(overflowFree)))
	l.Infof("Will prepare %d tasks", prepareCount)
	if prepareCount <= 0 {
		l.Infof("Skipping prepare, window is full")
		return nil
	}
	if err := c.pool.Restore(l); err != nil {
		return err
	}
	groupOrder := c.pool.GroupOrder()
	if len(groupOrder) == 0 {
		l.Infof("Skipping prepare, groups order empty")
		return nil
	}
	currentRecords := c.ws.Records()
	groups := make(map[string]bool)
	// Check that all hosts in one group. Skip muted tasks.
	for _, record := range currentRecords.SkipMuted() {
		groups[record.Task.Group] = true
	}
	if len(groups) > 1 {
		l.Warnf("Skipping prepare, many groups in working set: '%v'", groups)
		return nil
	}
	grName := groupOrder[0]
	if len(groups) == 1 {
		for gr := range groups {
			grName = gr
		}
	}
	group := c.pool.Group()[grName]
	l.Infof("Trying to fetch hosts from %s", grName)
	if len(group) == 0 {
		l.Infof("No hosts left in %s group, skipping prepare", grName)
		return nil
	}
	l.Infof("%d hosts left in %s group", len(group), grName)
	prepared := make([]*pb.WorkingSetRecord, 0, prepareCount)
	preparedHostnames := make([]string, 0, prepareCount)
	preparedServices := set{make(map[string]struct{})}
	ypInfo, err := c.ypStorage.Nodes()
	_ = preparedServices
	_ = ypInfo
	if err != nil {
		return err
	}
	for i := 0; i < len(group); i++ {
		if len(prepared) >= prepareCount {
			break
		}
		hostname := group[i]
		host, err := w.GetHost(hostname)
		if err != nil {
			return err
		}
		if slices.ContainsString(c.pool.Order, order.Movability) {
			if node, ok := ypInfo[hostname]; ok {
				buffService := []string{}
				var intersection bool
				for _, s := range node.NannyServices {
					if preparedServices.Contains(s) {
						intersection = true
						break
					} else {
						buffService = append(buffService, s)
					}
				}
				if intersection {
					continue
				} else {
					for _, v := range buffService {
						preparedServices.Add(v)
					}
				}
			}
		}
		if firmwareProblems, err := w.FirmwareProblems(hostname); err != nil {
			return err
		} else {
			host.FirmwareProblems = firmwareProblems
		}
		prj, err := w.GetProject(host.Project)
		if err != nil {
			return err
		}
		if err := validators.Valid(host, prj, c.j.Spec, c.j.Meta.Author, w); err != nil {
			l.Errorf("Validation failed for %s  on task preparing: %s", hostname, err)
			// TODO: [3/10/21] (vaspahomov): log err here, show status in ui
			continue
		}
		task := tasks.CreateTask(c.j.Spec.Action, grName, host)
		l.Infof("Prepared %s on %s", task.Meta.Action, task.Hostname)
		prepared = append(prepared, &pb.WorkingSetRecord{
			Task: task,
			Host: host,
		})
		preparedHostnames = append(preparedHostnames, hostname)
	}
	c.ws.Add(prepared...)
	c.pool.QueuePopHosts(preparedHostnames, grName)
	if err := c.pool.Dump(l); err != nil {
		return err
	}
	if err := c.ws.Dump(); err != nil {
		return err
	}
	return nil
}

// Starting prepared tasks
func (c *Controller) startTasks(w walle.IClient, l log.Logger) error {
	// Fast path if we do not need to start tasks
	if len(c.ws.Records().Created()) == 0 {
		return nil
	}
	for _, t := range c.ws.Records().Created() {
		host, err := w.GetHost(t.Task.Hostname)
		if err != nil {
			return err
		}
		prj, err := w.GetProject(host.Project)
		if err != nil {
			return err
		}
		if err := validators.Valid(host, prj, c.j.Spec, c.j.Meta.Author, w); err != nil {
			l.Errorf("Validation failed for %s on task staring: %s", t.Task.Hostname, err)
			// TODO: [3/10/21] (vaspahomov): log err here, show status in ui
			continue
		}
		l.Infof("Starting %s on %s", t.Task.Meta.Action, t.Task.Hostname)
		if err := tasks.StartTask(t.Task, c.j.Spec, l, w, c.dryRun, false); err != nil {
			return err
		}
		t.Task.State = pb.Task_RUNNING
	}
	return c.ws.Dump()
}

// Starting prepared tasks
func (c *Controller) cancelTasks(w walle.IClient, l log.Logger) error {
	needCancel := c.ws.Records().FilterEnforce([]pb.Task_Enforce{pb.Task_NEED_CANCEL})
	// Fast path if we do not need to cancel tasks
	if len(needCancel) == 0 {
		return nil
	}
	for _, t := range needCancel {
		l.Infof("Canceling task on %s", t.Task.Hostname)
		if !c.dryRun {
			err := w.CancelTask(t.Task.Hostname)
			if err != nil {
				return err
			}
		}
		t.Task.Enforce = pb.Task_CANCEL_WAITING
	}
	return c.ws.Dump()
}

// Starting prepared tasks
func (c *Controller) enforceTasks(w walle.IClient, l log.Logger) error {
	needEnforce := c.ws.Records().FilterEnforce([]pb.Task_Enforce{pb.Task_NEED_ENFORCE})
	// Fast path if we do not need to enforce tasks
	if len(needEnforce) == 0 {
		return nil
	}
	for _, t := range needEnforce {
		l.Infof("Starting %s with enforce on %s", t.Task.Meta.Action, t.Task.Hostname)
		err := tasks.StartTask(t.Task, c.j.Spec, l, w, c.dryRun, true)
		if err != nil {
			return err
		}
		t.Task.Enforce = pb.Task_ENFORCED
	}
	if err := c.ws.Dump(); err != nil {
		return err
	}
	names := make([]string, len(needEnforce))
	for i, t := range needEnforce {
		names[i] = t.Task.Hostname
	}
	if err := c.jobStorage.AppendEnforced(names); err != nil {
		return err
	}
	return nil
}

// pseudo set go structure
type set struct {
	value map[string]struct{}
}

func (s *set) Contains(m string) bool {
	if _, ok := s.value[m]; ok {
		return true
	} else {
		return false
	}
}

func (s *set) Add(m string) {
	s.value[m] = struct{}{}
}
