package job

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/golang/protobuf/ptypes"
	"go.etcd.io/etcd/clientv3"

	"a.yandex-team.ru/infra/maxwell/go/internal/pbutil"
	"a.yandex-team.ru/infra/maxwell/go/internal/storages"
	"a.yandex-team.ru/infra/maxwell/go/internal/validators"
	"a.yandex-team.ru/infra/maxwell/go/pkg/nanny"
	"a.yandex-team.ru/infra/maxwell/go/pkg/walle"
	"a.yandex-team.ru/infra/maxwell/go/pkg/yp"
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
)

func NewManager(client *clientv3.Client, inMemory bool, w walle.IClient,
	l log.Logger, dryRun bool, n nanny.Client, y yp.ClientPool) (*Manager, error) {
	return &Manager{
		etcd:           client,
		storage:        storages.NewJobs(inMemory, client),
		hostsStorage:   storages.NewHosts(inMemory, client),
		managerStorage: storages.NewManager(inMemory, client),
		ypStorage:      storages.NewYpNodes(inMemory, client),
		nannyStorage:   storages.NewNannyServices(inMemory, client),
		inMemory:       inMemory,
		w:              w,
		l:              l,
		jobsStorages:   make(JobsStorages),
		metrics:        make(map[string]*Metrics),
		dryRun:         dryRun,
		n:              n,
		y:              y,
	}, nil
}

type Manager struct {
	etcd           *clientv3.Client
	storage        storages.Jobs
	hostsStorage   storages.Hosts
	managerStorage storages.Manager
	jobsStorages   JobsStorages
	ypStorage      storages.YpNodes
	nannyStorage   storages.NannyServices
	w              walle.IClient
	l              log.Logger
	inMemory       bool
	dryRun         bool

	metrics map[string]*Metrics
	n       nanny.Client
	y       yp.ClientPool
}

type JobsStorages map[string]storages.Job

func (ps JobsStorages) Get(name string, create func() storages.Job) storages.Job {
	s, ok := ps[name]
	if !ok {
		s = create()
		ps[name] = s
	}
	return s
}

func (m *Manager) newStorage(name string) func() storages.Job {
	return func() storages.Job {
		return storages.NewJob(m.inMemory, m.etcd, name)
	}
}

func (m *Manager) RunMonitoring(ctx context.Context) error {
	m.l.Infof("Starting update manager")
	updateMetrics := time.NewTicker(time.Minute)
	for {
		select {
		case <-ctx.Done():
			m.l.Infof("Update manager stopped")
			if err := ctx.Err(); err != nil {
				m.l.Warnf("Context closed: %s", ctx.Err())
			}
			return nil
		case <-updateMetrics.C:
			m.l.Infof("Starting update metrics...")
			if err := m.updateMetrics(); err != nil {
				m.l.Errorf("%s", err)
				continue
			}
			m.l.Infof("Finished update metrics")
		}
	}
}

/*
Run starts job manager
Job Manager ticking:
* Refresh jobs working set (update hosts in working set, preparing/starting new tasks) **every minute**
* Update jobs pools. Refetching all hosts by source from spec **every 10minutes**
*/
func (m *Manager) RunMaster(ctx context.Context) error {
	m.l.Infof("Starting leader manager")
	updateTicker := time.NewTicker(time.Minute)
	updatePoolTicker := time.NewTicker(30 * time.Minute)
	updateCloudInfoTicker := time.NewTicker(10 * time.Minute)

	for {
		select {
		case <-ctx.Done():
			m.l.Infof("Leader manager stopped")
			if err := ctx.Err(); err != nil {
				m.l.Warnf("Context closed: %s", ctx.Err())
			}
			return nil
		case <-updateTicker.C:
			m.l.Infof("Refreshing working sets...")
			if err := m.refreshWorkingSets(); err != nil {
				m.l.Errorf("%s", err)
				continue
			}
			m.l.Infof("Finished refreshing working sets")
		case <-updatePoolTicker.C:
			m.l.Infof("Starting update pools...")
			if err := m.updatePools(); err != nil {
				m.l.Errorf("%s", err)
				continue
			}
			m.l.Infof("Finished update pools")
		case <-updateCloudInfoTicker.C:
			m.l.Infof("Starting update yp and nanny info...")
			if err := m.updateYpStorage(ctx); err != nil {
				m.l.Errorf("Failed to update yp: %s", err)
				continue
			}
			if err := m.updateNannyStorage(); err != nil {
				m.l.Errorf("Failed to update nanny: %s", err)
				continue
			}
			m.l.Infof("Finished refreshing yp and nanny data")
		}
	}
}

func (m *Manager) updatePools() error {
	jobs, err := m.storage.List()
	if err != nil {
		return err
	}
	for _, job := range jobs {
		jobStorage := m.jobsStorages.Get(job.Spec.Name, m.newStorage(job.Spec.Name))
		c := NewController(job, jobStorage, m.storage, m.hostsStorage, m.nannyStorage, m.ypStorage, m.dryRun)
		l := m.l.WithName(job.Spec.Name)
		l.Infof("Starting c.UpdatePool()")
		startTime := time.Now()
		err = c.UpdatePool(m.w, l)
		if err != nil {
			l.Errorf("Failed to update pool: %s", err)
		} else {
			l.Infof("Updated pool in %fs", time.Since(startTime).Seconds())
		}
	}
	return nil
}

func (m *Manager) updatePool(name string) error {
	job, err := m.storage.Get(name)
	if err != nil {
		return err
	}
	jobStorage := m.jobsStorages.Get(job.Spec.Name, m.newStorage(job.Spec.Name))
	c := NewController(job, jobStorage, m.storage, m.hostsStorage, m.nannyStorage, m.ypStorage, m.dryRun)
	l := m.l.WithName(job.Spec.Name)
	startTime := time.Now()
	err = c.UpdatePool(m.w, l)
	if err != nil {
		l.Errorf("Failed to update %s pool: %s", job.Spec.Name, err)
	} else {
		l.Infof("Updated pool in %fs", time.Since(startTime).Seconds())
	}
	return nil
}

func (m *Manager) Master() (string, error) {
	return m.managerStorage.Master()
}

func (m *Manager) refreshWorkingSets() error {
	jobs, err := m.storage.List()
	if err != nil {
		return err
	}
	names := make([]string, len(jobs))
	for i, job := range jobs {
		names[i] = job.Spec.Name
	}
	m.l.Infof("Going to update [%s]...", strings.Join(names, ","))
	for _, job := range jobs {
		m.l.Infof("Starting update %s...", job.Spec.Name)
		jobStorage := m.jobsStorages.Get(job.Spec.Name, m.newStorage(job.Spec.Name))
		c := NewController(job, jobStorage, m.storage, m.hostsStorage, m.nannyStorage, m.ypStorage, m.dryRun)
		err = c.RefreshWorkingSet(m.w, m.l)
		if err != nil {
			m.l.Errorf("Failed to update %s: %s", job.Spec.Name, err)
			continue
		}
	}
	return nil
}

func (m *Manager) updateMetrics() error {
	names, err := m.storage.Names()
	if err != nil {
		return err
	}
	for _, name := range names {
		mtrcs, ok := m.metrics[name]
		if !ok {
			mtrcs = NewMetrics(name)
			mtrcs.Register()
			m.metrics[name] = mtrcs
		}
		job, err := m.Get(name)
		if err != nil {
			return err
		}
		if job == nil {
			delete(m.metrics, name)
		} else {
			NewMonitoring(job).UnistatUpdate(mtrcs)
		}
	}
	return nil
}

// Put job create/update job from given spec.
// After this action running condition will be setted to False
func (m *Manager) PutJob(spec *pb.Job_Spec, owner string) error {
	m.l.Infof("Putting %s job spec...", spec.Name)
	if err := validators.SpecIsValid(spec); err != nil {
		return err
	}
	j, err := m.storage.Get(spec.Name)
	if err != nil {
		return err
	}
	if j == nil {
		job, err := m.CreateJob(spec, owner)
		if err != nil {
			return err
		}
		j = job
	} else {
		if err := m.UpdateJob(j, spec, owner); err != nil {
			return err
		}
	}
	if err := m.storage.Put(j); err != nil {
		return err
	}
	return m.updatePool(spec.Name)
}

func (m *Manager) CreateJob(spec *pb.Job_Spec, owner string) (*pb.Job, error) {
	m.l.Infof("Creating new job %s...", spec.Name)
	creationTime := ptypes.TimestampNow()
	j := &pb.Job{
		Meta: &pb.Job_Meta{
			Author:       owner,
			CreationTime: creationTime,
			Mtime:        creationTime,
		},
		Spec: spec,
		Status: &pb.Job_Status{
			Running: &pb.Condition{},
			Sleep:   &pb.Condition{},
		},
	}
	status := j.Status
	// Do not start job right after creation
	pbutil.SetFalse(status.Running, "Paused after creation")
	pbutil.SetFalse(status.Sleep, "")
	return j, nil
}

func (m *Manager) UpdateJob(j *pb.Job, spec *pb.Job_Spec, owner string) error {
	m.l.Infof("Updating %s job spec...", spec.Name)
	j.Meta.Mtime = ptypes.TimestampNow()
	j.Meta.Author = owner
	j.Spec = spec
	status := j.Status
	// Set on pause after update
	pbutil.SetFalse(status.Running, "Paused after update")
	return nil
}

func (m *Manager) Update(job *pb.Job) error {
	j, err := m.storage.Get(job.Spec.Name)
	if err != nil {
		return err
	}
	if j == nil {
		return fmt.Errorf("'%s' job not found", job.Spec.Name)
	}
	if err := m.storage.Put(job); err != nil {
		return err
	}
	return nil
}

func (m *Manager) List() ([]*pb.Job, error) {
	return m.storage.List()
}

func (m *Manager) Get(name string) (*pb.FullJob, error) {
	j, err := m.storage.Get(name)
	if err != nil {
		return nil, err
	}
	jobStorage := m.jobsStorages.Get(j.Spec.Name, m.newStorage(j.Spec.Name))
	c := NewController(j, jobStorage, m.storage, m.hostsStorage, m.nannyStorage, m.ypStorage, m.dryRun)
	if err := c.pool.Restore(m.l); err != nil {
		return nil, err
	}
	if err := c.ws.Restore(); err != nil {
		return nil, err
	}
	processed, err := jobStorage.Processed()
	if err != nil {
		return nil, err
	}
	enforced, err := jobStorage.Enforced()
	if err != nil {
		return nil, err
	}
	groupOrder := c.pool.GroupOrder()
	groups := c.pool.Group()
	gr := make([]*pb.FullJob_Queue, len(groupOrder))
	for i, groupName := range c.pool.GroupOrder() {
		gr[i] = &pb.FullJob_Queue{
			Group: groupName,
			Hosts: groups[groupName],
		}
	}
	return &pb.FullJob{
		Job:        j,
		WorkingSet: c.ws.Records(),
		Groups:     gr,
		State: &pb.JobState{
			Processed: processed,
			Enforced:  enforced,
		},
	}, nil
}

// MuteTask for job. Muted tasks will not counted in processing tasks (tasks in working set)
func (m *Manager) MuteTask(jobName, hostname, author string) error {
	j, err := m.storage.Get(jobName)
	if err != nil {
		return err
	}
	jobStorage := m.jobsStorages.Get(j.Spec.Name, m.newStorage(j.Spec.Name))
	c := NewController(j, jobStorage, m.storage, m.hostsStorage, m.nannyStorage, m.ypStorage, m.dryRun)
	return c.MuteTask(hostname, author)
}

/*
Completely removing job
* Stop managing
* Remove prepared hosts queue in pool
* Remove working set
* Remove processed hosts
*/
func (m *Manager) Delete(name string) error {
	if err := m.jobsStorages.Get(name, m.newStorage(name)).Delete(); err != nil {
		return err
	}
	return m.storage.Delete(name)
}

func (m *Manager) Summary(name string) (*pb.Job, error) {
	j, err := m.storage.Get(name)
	if err != nil {
		return nil, err
	}
	return j, nil
}

// refreshYPInfo fetch and consolidate yp nodes data from all yp clusters
func (m *Manager) refreshYPInfo(ctx context.Context) (map[string]*pb.YpNode, error) {
	nodes := make(map[string]*pb.YpNode)
	for _, c := range yp.YpClusters {
		yc := m.y[c]
		pods, err := yc.GetPods(ctx)
		if err != nil {
			m.l.Errorf("got error in fetching yp pods, error: %s", err)
			return nil, err
		}
		podsets, err := yc.GetPodSets(ctx)
		if err != nil {
			m.l.Errorf("got error in fetching yp podsets, error: %s", err)
			return nil, err
		}
		budgets, err := yc.GetBudgets(ctx)
		if err != nil {
			m.l.Errorf("got error in fetching yp budgets, error: %s", err)
			return nil, err
		}
		// pdbs is a pod disruption budget
		pdbs := yp.GetPodsetPdbs(podsets, budgets)
		clusterNodes := yp.CompileNodes(pods, pdbs)
		for k, v := range clusterNodes {
			nodes[k] = v
		}
		m.l.Infof("Fetched info for %s yp cluster", c)

	}
	return nodes, nil
}

// refreshNanny fetch and processed nanny data
func (m *Manager) refreshNannyInfo() (map[string]*pb.NannyService, error) {
	summaries, err := m.n.ListSummaries()
	if err != nil {
		m.l.Errorf("got error in fetching nanny summaries, error: %s", err)
		return nil, err
	}
	m.l.Infof("listed %d nanny replication summaries", len(summaries))
	policies, err := m.n.ListReplicationPolicies()
	m.l.Infof("listed %d nanny replication policies", len(policies))
	if err != nil {
		m.l.Errorf("got error in fetching nanny policies %s", err)
		return nil, err
	}
	return nanny.CompileServices(policies, summaries), nil
}

func (m *Manager) updateYpStorage(ctx context.Context) error {
	nodes, err := m.refreshYPInfo(ctx)
	if err != nil {
		m.l.Errorf("got errors when updating yp info: %s", err)
		return err
	}
	return m.ypStorage.PutNodes(nodes)
}

func (m *Manager) updateNannyStorage() error {
	services, err := m.refreshNannyInfo()
	if err != nil {
		m.l.Errorf("got errors when updating nanny info: %s", err)
		return err
	}
	return m.nannyStorage.PutServices(services)
}
