package job

import (
	"a.yandex-team.ru/infra/maxwell/go/pkg/metrics"
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/yandex/unistat"
)

func NewMonitoring(job *pb.FullJob) *Monitoring {
	return &Monitoring{job: job}
}

type Monitoring struct {
	job *pb.FullJob
}

func (m *Monitoring) ProcessedHosts() int {
	return len(m.job.State.Processed)
}

func (m *Monitoring) EnforcedHosts() int {
	return len(m.job.State.Enforced)
}

func (m *Monitoring) Remaining() int {
	remaining := 0
	for _, group := range m.job.Groups {
		remaining += len(group.Hosts)
	}
	return remaining
}

func (m *Monitoring) Running() int {
	running := 0
	for _, r := range m.job.WorkingSet {
		if r.Task.State == pb.Task_RUNNING {
			running += 1
		}
	}
	return running
}

func (m *Monitoring) Overflow() int {
	running := 0
	for _, r := range m.job.WorkingSet {
		if r.Task.State == pb.Task_WAITING {
			running += 1
		}
	}
	return running
}

func (m *Monitoring) Window() int {
	return int(m.job.Job.Spec.Window)
}

func (m *Monitoring) WindowOverflow() int {
	return int(m.job.Job.Spec.WindowOverflow)
}

func (m *Monitoring) UnistatUpdate(mtrcs *Metrics) {
	mtrcs.running.Update(float64(m.Running()))
	mtrcs.remaining.Update(float64(m.Remaining()))
	mtrcs.processed.Update(float64(m.ProcessedHosts()))
	mtrcs.enforced.Update(float64(m.EnforcedHosts()))
	mtrcs.overflow.Update(float64(m.Overflow()))
	mtrcs.window.Update(float64(m.Window()))
	mtrcs.windowOverflow.Update(float64(m.WindowOverflow()))
}

func NewMetrics(name string) *Metrics {
	tags := map[string]string{"tier": name}
	last := unistat.StructuredAggregation{
		AggregationType: unistat.Absolute,
		Group:           unistat.Last,
		MetaGroup:       unistat.Last,
		Rollup:          unistat.Last,
	}
	return &Metrics{
		enforced:       unistat.NewNumeric(metrics.AppendTags("maxwell-enforced", tags), 50, last, unistat.Last),
		processed:      unistat.NewNumeric(metrics.AppendTags("maxwell-processed", tags), 50, last, unistat.Last),
		remaining:      unistat.NewNumeric(metrics.AppendTags("maxwell-remaining", tags), 50, last, unistat.Last),
		running:        unistat.NewNumeric(metrics.AppendTags("maxwell-running", tags), 50, last, unistat.Last),
		overflow:       unistat.NewNumeric(metrics.AppendTags("maxwell-overflow", tags), 50, last, unistat.Last),
		window:         unistat.NewNumeric(metrics.AppendTags("maxwell-window", tags), 50, last, unistat.Last),
		windowOverflow: unistat.NewNumeric(metrics.AppendTags("maxwell-window-overflow", tags), 50, last, unistat.Last),
	}
}

func (m *Metrics) Register() {
	unistat.Register(m.enforced)
	unistat.Register(m.processed)
	unistat.Register(m.remaining)
	unistat.Register(m.running)
	unistat.Register(m.overflow)
	unistat.Register(m.window)
	unistat.Register(m.windowOverflow)
}

type Metrics struct {
	processed      *unistat.Numeric
	enforced       *unistat.Numeric
	remaining      *unistat.Numeric
	running        *unistat.Numeric
	overflow       *unistat.Numeric
	window         *unistat.Numeric
	windowOverflow *unistat.Numeric
}
