// Source: https://a.yandex-team.ru/arc/trunk/arcadia/infra/nanny2/pkg/hq/housekeeping/leader.go
package leader

import (
	"a.yandex-team.ru/library/go/core/log"
	"context"
	"go.etcd.io/etcd/clientv3"
	"go.etcd.io/etcd/clientv3/concurrency"
	"time"
)

type CancellableFunc func(context.Context)

func notFound(key string) clientv3.Cmp {
	return clientv3.Compare(clientv3.ModRevision(key), "=", 0)
}

func LockAndRun(ctx context.Context, l log.Logger, c *clientv3.Client, name, id string, runnable CancellableFunc) {
	l.Infof("Started locking %s", name)
	for {
	session:
		cancelCtx, cancelFunc := context.WithCancel(ctx)
		l.Info("Creating session...")
		s, err := concurrency.NewSession(c)
		if err == nil {
			l.Infof("Created session with lease ID: %d", s.Lease())
			for {
				l.Debug("Iterating")
				txnResp, err := c.KV.Txn(ctx).If(
					notFound(name),
				).Then(
					clientv3.OpPut(name, id, clientv3.WithLease(s.Lease())),
				).Commit()
				if err != nil {
					if ctx.Err() != nil {
						// We have been cancelled
						cancelFunc()
						return
					}
					// We might have disconnected and need to start a new session
					l.Errorf("Failed to create lock file: %s", err.Error())
					s.Orphan()
					time.Sleep(5 * time.Second) // Avoid busy looping
					goto session
				} else if !txnResp.Succeeded {
					l.Debug("Transaction failed, will retry...")
					time.Sleep(5 * time.Second)
				} else {
					break
				}
			}
			// Now we have created file with lease, assuming lock is held
			l.Infof("Assuming control over '%s', starting runnable", name)
			go runnable(cancelCtx)
			select {
			case <-ctx.Done(): // If initial context is Done - stop all activities
				l.Info("Lock and run cancelled, stop runnable...")
				cancelFunc()
				return
			case <-s.Done(): // If session is Done, just start all over again
				l.Info("Session is done, stop runnable...")
				cancelFunc()
				break
			}
			l.Info("Stopped runnable, will retry")
			continue
		} else if ctx.Err() != nil {
			l.Infof("Context cancelled %s", ctx.Err().Error())
		} else {
			l.Errorf("Failed to establish session: %s", err.Error())
			time.Sleep(5 * time.Second)
		}
	}
}
