package filter

import (
	"testing"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/test/assertpb"
)

var (
	automationProjects = map[string]*pb.Project{"enabled": {
		HealingAutomation: pb.Project_ENABLED,
		DnsAutomation:     pb.Project_ENABLED,
	}, "disabled-dns": {
		HealingAutomation: pb.Project_ENABLED,
		DnsAutomation:     pb.Project_DISABLED,
	}, "disabled-healing": {
		HealingAutomation: pb.Project_DISABLED,
		DnsAutomation:     pb.Project_ENABLED,
	}, "disabled": {
		HealingAutomation: pb.Project_DISABLED,
		DnsAutomation:     pb.Project_DISABLED,
	}}
	enabledAutomationHost = &pb.Host{
		Project:  "enabled",
		Hostname: "enabled",
	}
	disabledDNSHost = &pb.Host{
		Project:  "disabled-dns",
		Hostname: "disabled-dns",
	}
	disabledHealingHost = &pb.Host{
		Project:  "disabled-healing",
		Hostname: "disabled-healing",
	}
	disabledAutomationHost = &pb.Host{
		Project:  "disabled",
		Hostname: "disabled",
	}
)

func TestAutomation_Filter(t *testing.T) {
	tests := []struct {
		name     string
		before   []*pb.Host
		after    []*pb.Host
		projects map[string]*pb.Project
		s        *pb.Filter_Automation
		err      string
	}{{
		name: "filter disabled all automation host",
		before: []*pb.Host{
			disabledAutomationHost,
		},
		after:    make([]*pb.Host, 0),
		projects: automationProjects,
		s:        &pb.Filter_Automation{Enabled: true},
	}, {
		name: "filter disabled dns automation host",
		before: []*pb.Host{
			disabledDNSHost,
		},
		after:    make([]*pb.Host, 0),
		projects: automationProjects,
		s:        &pb.Filter_Automation{Enabled: true},
	}, {
		name: "filter disabled healing automation host",
		before: []*pb.Host{
			disabledHealingHost,
		},
		after:    make([]*pb.Host, 0),
		projects: automationProjects,
		s:        &pb.Filter_Automation{Enabled: true},
	}, {
		name: "filter different hosts",
		before: []*pb.Host{
			enabledAutomationHost,
			disabledAutomationHost,
			disabledHealingHost,
			disabledDNSHost,
		},
		after: []*pb.Host{
			enabledAutomationHost,
		},
		projects: automationProjects,
		s:        &pb.Filter_Automation{Enabled: true},
	}, {
		name: "skip filter",
		before: []*pb.Host{
			enabledAutomationHost,
			disabledAutomationHost,
			disabledHealingHost,
			disabledDNSHost,
		},
		after: []*pb.Host{
			enabledAutomationHost,
			disabledAutomationHost,
			disabledHealingHost,
			disabledDNSHost,
		},
		projects: automationProjects,
		s:        &pb.Filter_Automation{Enabled: false},
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			r := &Automation{
				s: tt.s,
			}
			got := r.Filter(tt.before, tt.projects, l)
			if len(got) != len(tt.after) {
				t.Errorf("len(before) != len(after) | %d != %d", len(tt.before), len(tt.after))
				return
			}
			for i := 0; i < len(got); i++ {
				gotHost := got[i]
				expectedHost := tt.after[i]
				assertpb.Equal(t, expectedHost, gotHost)
			}
		})
	}
}
