package filter

import (
	"testing"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/test/assertpb"
)

func TestRestrictions_Filter(t *testing.T) {
	restrictionsProjects := map[string]*pb.Project{
		"automated-reboot": {DefaultRestrictions: []string{"automated-reboot"}},
		"reboot":           {DefaultRestrictions: []string{"automated-reboot", "reboot"}},
		"no":               {DefaultRestrictions: make([]string, 0)},
	}
	restrictRebootFromProjectHost := &pb.Host{
		Restrictions: make([]string, 0),
		Project:      "reboot",
		Hostname:     "reboot-from-project",
	}
	restrictAutomatedRebootFromProjectHost := &pb.Host{
		Restrictions: make([]string, 0),
		Project:      "automated-reboot",
		Hostname:     "automated-reboot-from-project",
	}
	restrictAutomatedRebootHost := &pb.Host{
		Restrictions: []string{"automated-reboot"},
		Project:      "no",
		Hostname:     "automated-reboot",
	}
	restrictRebootHost := &pb.Host{
		Restrictions: []string{"automated-reboot", "reboot"},
		Project:      "no",
		Hostname:     "reboot",
	}
	restrictRebootComplexHost := &pb.Host{
		Restrictions: []string{"automated-reboot", "reboot"},
		Project:      "reboot",
		Hostname:     "complex-reboot",
	}
	withoutRestrictionHost := &pb.Host{
		Project:  "no",
		Hostname: "no",
	}
	tests := []struct {
		name     string
		before   []*pb.Host
		after    []*pb.Host
		projects map[string]*pb.Project
		s        *pb.Filter_Restrictions
		err      string
	}{{
		name: "filter automated reboot restriction",
		before: []*pb.Host{
			restrictRebootFromProjectHost,
			restrictAutomatedRebootFromProjectHost,
			restrictAutomatedRebootHost,
			restrictRebootHost,
			restrictRebootComplexHost,
			withoutRestrictionHost,
		},
		after: []*pb.Host{
			withoutRestrictionHost,
		},
		projects: restrictionsProjects,
		s: &pb.Filter_Restrictions{
			Check: []string{"automated-reboot"},
		},
	}, {
		name: "filter reboot restriction",
		before: []*pb.Host{
			restrictRebootFromProjectHost,
			restrictAutomatedRebootFromProjectHost,
			restrictAutomatedRebootHost,
			restrictRebootHost,
			restrictRebootComplexHost,
			withoutRestrictionHost,
		},
		after: []*pb.Host{
			restrictAutomatedRebootFromProjectHost,
			restrictAutomatedRebootHost,
			withoutRestrictionHost,
		},
		projects: restrictionsProjects,
		s: &pb.Filter_Restrictions{
			Check: []string{"reboot"},
		},
	}, {
		name: "filter reboot and automated-reboot restriction",
		before: []*pb.Host{
			restrictRebootFromProjectHost,
			restrictAutomatedRebootFromProjectHost,
			restrictAutomatedRebootHost,
			restrictRebootHost,
			restrictRebootComplexHost,
			withoutRestrictionHost,
		},
		after: []*pb.Host{
			withoutRestrictionHost,
		},
		projects: restrictionsProjects,
		s: &pb.Filter_Restrictions{
			Check: []string{"reboot", "automated-reboot"},
		},
	}, {
		name: "no restrictions",
		before: []*pb.Host{
			restrictRebootFromProjectHost,
			restrictAutomatedRebootFromProjectHost,
			restrictAutomatedRebootHost,
			restrictRebootHost,
			restrictRebootComplexHost,
			withoutRestrictionHost,
		},
		after: []*pb.Host{
			restrictRebootFromProjectHost,
			restrictAutomatedRebootFromProjectHost,
			restrictAutomatedRebootHost,
			restrictRebootHost,
			restrictRebootComplexHost,
			withoutRestrictionHost,
		},
		projects: restrictionsProjects,
		s: &pb.Filter_Restrictions{
			Check: make([]string, 0),
		},
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			r := &Restrictions{
				s: tt.s,
			}
			got := r.Filter(tt.before, tt.projects, l)
			if len(got) != len(tt.after) {
				t.Errorf("len(before) != len(after) | %d != %d", len(tt.before), len(tt.after))
			}
			for i := 0; i < len(got); i++ {
				gotHost := got[i]
				expectedHost := tt.after[i]
				assertpb.Equal(t, expectedHost, gotHost)
			}
		})
	}
}
