package order

import (
	"sort"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

type LessFunc func(a, b *pb.Host, nanny map[string]*pb.NannyService, yp map[string]*pb.YpNode) bool

type SortableHosts struct {
	hosts []*pb.Host
	less  LessFunc
	nanny map[string]*pb.NannyService
	yp    map[string]*pb.YpNode
}

func (pq *SortableHosts) Len() int { return len(pq.hosts) }

func (pq *SortableHosts) Less(i, j int) bool {
	return pq.less(pq.hosts[i], pq.hosts[j], pq.nanny, pq.yp)
}

func (pq *SortableHosts) Swap(i, j int) {
	pq.hosts[i], pq.hosts[j] = pq.hosts[j], pq.hosts[i]
}

func Order(hosts []*pb.Host, orders []string, nanny map[string]*pb.NannyService, yp map[string]*pb.YpNode) []string {
	sortable := &SortableHosts{
		hosts: make([]*pb.Host, len(hosts)),
		nanny: nanny,
		yp:    yp,
		less:  less(orders),
	}
	// Do not reorder source slice
	copy(sortable.hosts, hosts)
	sort.Stable(sortable)
	hostnames := make([]string, len(hosts))
	for i, h := range sortable.hosts {
		hostnames[i] = h.Hostname
	}
	return hostnames
}

const (
	LocationOrder = "location"
	RackOrder     = "rack"
	HostnameOrder = "hostname"
	Movability    = "movability"
)

func less(orders []string) LessFunc {
	ordersSet := make(map[string]bool)
	for _, order := range orders {
		ordersSet[order] = true
	}
	return func(a, b *pb.Host, nanny map[string]*pb.NannyService, yp map[string]*pb.YpNode) bool {
		if ordersSet[Movability] {
			return SortByMovability(MovabilityProps(a.Hostname, nanny, yp), MovabilityProps(b.Hostname, nanny, yp))
		}
		if ordersSet[LocationOrder] {
			c := Dc(a.Location, b.Location)
			if c != 0 {
				return c < 0
			}
		}
		if ordersSet[RackOrder] {
			c := Rack(a.Location, b.Location)
			if c != 0 {
				return c < 0
			}
		}
		if ordersSet[HostnameOrder] {
			c := Hostname(a.Hostname, b.Hostname)
			if c != 0 {
				return c < 0
			}
		}
		return false
	}
}

type hostCloudProps struct {
	budgetMinimum int
	Movable       bool
}

func MovabilityProps(Hostname string, nanny map[string]*pb.NannyService, yp map[string]*pb.YpNode) *hostCloudProps {
	props := &hostCloudProps{}
	if ypinfo, ok := yp[Hostname]; ok {
		props.budgetMinimum = int(ypinfo.BudgetMinimum)
		props.Movable = true
		for _, service := range ypinfo.NannyServices {
			if n, ok := nanny[service]; ok {
				if !n.Moveable {
					props.Movable = false
					break
				}
			}
		}

	}
	return props
}

func SortByMovability(props1 *hostCloudProps, props2 *hostCloudProps) bool {
	if (props2.Movable && props1.Movable) || (!props2.Movable && !props1.Movable) {
		return props1.budgetMinimum > props2.budgetMinimum
	} else if props2.Movable && !props1.Movable {
		return false
	}
	return true
}
