package order

import (
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
	"github.com/stretchr/testify/assert"
	"testing"
)

func TestOrder(t *testing.T) {
	nanny := map[string]*pb.NannyService{}
	yp := map[string]*pb.YpNode{}
	tests := []struct {
		name    string
		hosts   []*pb.Host
		orders  []string
		ordered []string
	}{{
		name: "hostname ordered",
		hosts: []*pb.Host{
			{Hostname: "sas1-0002.search.yandex.net"},
			{Hostname: "sas1-0001.search.yandex.net"},
		},
		orders:  []string{HostnameOrder},
		ordered: []string{"sas1-0001.search.yandex.net", "sas1-0002.search.yandex.net"},
	}, {
		name: "rack ordered",
		hosts: []*pb.Host{
			{Hostname: "sas1-0001.search.yandex.net", Location: &pb.Host_Location{Rack: "2"}},
			{Hostname: "sas1-0002.search.yandex.net", Location: &pb.Host_Location{Rack: "1"}},
		},
		orders:  []string{RackOrder},
		ordered: []string{"sas1-0002.search.yandex.net", "sas1-0001.search.yandex.net"},
	}, {
		name: "rack/hostname ordered",
		hosts: []*pb.Host{
			{Hostname: "sas1-0001.search.yandex.net", Location: &pb.Host_Location{Rack: "2"}},
			{Hostname: "sas1-0002.search.yandex.net", Location: &pb.Host_Location{Rack: "1"}},
			{Hostname: "sas1-0003.search.yandex.net", Location: &pb.Host_Location{Rack: "1"}},
		},
		orders:  []string{RackOrder, HostnameOrder},
		ordered: []string{"sas1-0002.search.yandex.net", "sas1-0003.search.yandex.net", "sas1-0001.search.yandex.net"},
	}, {
		name: "hostname/rack ordered",
		hosts: []*pb.Host{
			{Hostname: "sas1-0001.search.yandex.net", Location: &pb.Host_Location{Rack: "2"}},
			{Hostname: "sas1-0002.search.yandex.net", Location: &pb.Host_Location{Rack: "1"}},
			{Hostname: "sas1-0003.search.yandex.net", Location: &pb.Host_Location{Rack: "1"}},
		},
		orders:  []string{HostnameOrder, RackOrder},
		ordered: []string{"sas1-0002.search.yandex.net", "sas1-0003.search.yandex.net", "sas1-0001.search.yandex.net"},
	}, {
		name: "hostname/rack ordered",
		hosts: []*pb.Host{
			{Hostname: "sas1-0001.search.yandex.net", Location: &pb.Host_Location{Country: "RU", City: "SAS"}},
			{Hostname: "sas1-0002.search.yandex.net", Location: &pb.Host_Location{Country: "FI", City: "MANSTALA"}},
			{Hostname: "sas1-0003.search.yandex.net", Location: &pb.Host_Location{Country: "FI", City: "MANSTALA"}},
		},
		orders:  []string{HostnameOrder, LocationOrder},
		ordered: []string{"sas1-0002.search.yandex.net", "sas1-0003.search.yandex.net", "sas1-0001.search.yandex.net"},
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got := Order(tt.hosts, tt.orders, nanny, yp)
			assert.Equal(t, tt.ordered, got)
		})
	}
}

func TestOrder2(t *testing.T) {
	test := SortableHosts{
		// checking that order swaps because first hosts has not moveable service
		hosts: []*pb.Host{{Hostname: "sas-test001.search.yandex.net"}, {Hostname: "sas-test002.search.yandex.net"}},
		nanny: map[string]*pb.NannyService{"nannytest1": {MetaId: "nannytest1", Moveable: false}, "nannytest2": {MetaId: "nannytest2", Moveable: true}, "nannytest3": {MetaId: "nannytest3", Moveable: true}},
		yp:    map[string]*pb.YpNode{"sas-test001.search.yandex.net": &pb.YpNode{MetaId: "sas-test001.search.yandex.net", BudgetMinimum: 6, NannyServices: []string{"nannytest1", "nannytest2"}}, "sas-test002.search.yandex.net": &pb.YpNode{MetaId: "sas-test002.search.yandex.net", BudgetMinimum: 4, NannyServices: []string{"nannytest2", "nannytest3"}}}}
	assert.Equal(t, Order(test.hosts, []string{Movability}, test.nanny, test.yp), []string{"sas-test002.search.yandex.net", "sas-test001.search.yandex.net"})

}
