package pool

import (
	"time"

	"a.yandex-team.ru/infra/maxwell/go/internal/pool/filter"
	"a.yandex-team.ru/infra/maxwell/go/internal/pool/group"
	"a.yandex-team.ru/infra/maxwell/go/internal/pool/order"
	"a.yandex-team.ru/infra/maxwell/go/internal/pool/source"
	"a.yandex-team.ru/infra/maxwell/go/internal/storages"
	"a.yandex-team.ru/infra/maxwell/go/pkg/walle"
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
)

func New(spec *pb.Job_Spec, jobStorage storages.Job, nannyStorage storages.NannyServices, ypStorage storages.YpNodes) *Pool {
	s := source.NewSource(spec.Source)
	filters := make([]filter.Filter, 0)
	filters = append(filters, filter.NewHealth(spec.Filters.Health))
	filters = append(filters, filter.NewRestrictions(spec.Filters.Restrictions))
	filters = append(filters, filter.NewAutomation(spec.Filters.Automation))
	return &Pool{
		source:       s,
		filters:      filters,
		Order:        spec.Order,
		group:        spec.Group,
		jobStorage:   jobStorage,
		nannyStorage: nannyStorage,
		ypStorage:    ypStorage,
		tmp:          &tmp{},
	}
}

type Pool struct {
	jobStorage   storages.Job
	nannyStorage storages.NannyServices
	ypStorage    storages.YpNodes

	// Fields from job spec
	source  source.Source
	filters []filter.Filter
	Order   []string
	group   []string

	*tmp
}

type tmp struct {
	groupOrder []string
	grouped    map[string][]string
}

func (p *Pool) Restore(l log.Logger) error {
	l.Infof("Restoring pool data from storage...")
	groups, err := p.jobStorage.Groups()
	if err != nil {
		l.Errorf("Failed to get groups: %s", err)
		return err
	}
	groupsOrder, err := p.jobStorage.GroupsOrder()
	if err != nil {
		l.Errorf("Failed to get groups order: %s", err)
		return err
	}
	p.grouped = groups
	p.groupOrder = groupsOrder
	return nil
}

func (p *Pool) Dump(l log.Logger) error {
	l.Infof("Persisting pool data...")
	if err := p.jobStorage.PutGroups(p.grouped); err != nil {
		l.Errorf("Failed to put groups: %s", err)
		return err
	}
	if err := p.jobStorage.PutGroupsOrder(p.groupOrder); err != nil {
		l.Errorf("Failed to put groups order: %s", err)
		return err
	}
	return nil
}
func (p *Pool) Group() map[string][]string {
	return p.grouped
}

func (p *Pool) QueuePopHosts(hostnames []string, groupName string) {
	if _, ok := p.grouped[groupName]; !ok {
		return
	}
	if len(p.grouped[groupName]) == len(hostnames) {
		delete(p.grouped, groupName)
		j := 0
		for i := 0; i < len(p.groupOrder); i++ {
			if p.groupOrder[i] != groupName {
				p.groupOrder[j] = p.groupOrder[i]
				j++
			}
		}
		p.groupOrder = p.groupOrder[:j]
	} else {
		j := 0
		for i := 0; i < len(p.grouped[groupName]); i++ {
			hostname := p.grouped[groupName][i]
			if !slices.ContainsString(hostnames, hostname) {
				p.grouped[groupName][j] = p.grouped[groupName][i]
				j++
			}
		}
		p.grouped[groupName] = p.grouped[groupName][:j]
	}
}

func (p *Pool) GroupOrder() []string {
	return p.groupOrder
}

func (p *Pool) Update(w walle.IClient, l log.Logger, skip []string) error {
	// yp nanny data sync from storage
	startTime := time.Now()
	l.Infof("Starting nanny.Services()")
	nanny, err := p.nannyStorage.Services()
	if err != nil {
		return err
	}
	l.Infof("Finished nanny.Services() in %fs", time.Since(startTime).Seconds())
	startTime = time.Now()
	l.Infof("Starting yp.Nodes()")
	yp, err := p.ypStorage.Nodes()
	if err != nil {
		return err
	}
	l.Infof("Finished yp.Nodes() in %fs", time.Since(startTime).Seconds())
	startTime = time.Now()
	l.Infof("Starting source.Hosts()")
	hosts, projects, err := p.source.Hosts(w, l)
	if err != nil {
		return err
	}
	l.Infof("Finished source.Hosts() in %fs", time.Since(startTime).Seconds())
	startTime = time.Now()
	l.Infof("Starting hosts filtering")
	for _, f := range p.filters {
		hosts = f.Filter(hosts, projects, l)
	}
	l.Infof("Finished hosts filtering in %fs", time.Since(startTime).Seconds())
	startTime = time.Now()
	l.Infof("Starting hosts ordering")
	var ordered []string
	// Use order from source when ordering omitted in spec
	if len(p.Order) == 0 {
		ordered = make([]string, len(hosts))
		for i, h := range hosts {
			ordered[i] = h.Hostname
		}
	} else {
		ordered = order.Order(hosts, p.Order, nanny, yp)
	}
	l.Infof("Finished hosts ordering in %fs", time.Since(startTime).Seconds())
	startTime = time.Now()
	l.Infof("Starting hosts filterNames")
	filtered := p.filterNames(ordered, skip, l)
	l.Infof("Finished hosts filterNames in %fs", time.Since(startTime).Seconds())
	startTime = time.Now()
	l.Infof("Starting hosts grouping")
	// Create single group on grouping omitted in spec
	if len(p.group) == 0 {
		p.grouped = map[string][]string{"single": ordered}
		p.groupOrder = []string{"single"}
	} else {
		p.grouped, p.groupOrder = group.Group(hosts, filtered, p.group)
	}
	l.Infof("Finished hosts grouping in %fs", time.Since(startTime).Seconds())
	return nil
}

func (p *Pool) filterNames(source, skip []string, l log.Logger) []string {
	filtered := make([]string, len(source))
	j := 0
	for i := 0; i < len(source); i++ {
		hostname := source[i]
		if !slices.ContainsString(skip, hostname) {
			filtered[j] = hostname
			j++
		} else {
			l.Infof("Not appending '%s' in queue, because explicitly skip", hostname)
		}
	}
	return filtered[:j]
}
