package source

import (
	"a.yandex-team.ru/infra/maxwell/go/pkg/walle"
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"github.com/stretchr/testify/mock"
	"reflect"
	"testing"
)

func TestWalle(t *testing.T) {
	type prepareMock func(w *walle.Fake)

	prestableReadyHost := &pb.Host{
		Health:           make(map[string]string),
		Location:         &pb.Host_Location{Country: "FI", City: "MANTSALA", Rack: "XZ"},
		Restrictions:     make([]string, 0),
		FirmwareProblems: make([]string, 0),
		Project:          "rtc-prestable",
		Hostname:         "man1-4537.search.yandex.net",
		Status:           "ready",
	}
	prestableProject := &pb.Project{
		DefaultRestrictions: make([]string, 0),
		HealingAutomation:   pb.Project_ENABLED,
		DnsAutomation:       pb.Project_ENABLED,
	}

	tests := []struct {
		name    string
		source  *pb.Source_Walle
		prepare prepareMock
		want    []*pb.Host
		want1   map[string]*pb.Project
		wantErr bool
	}{{
		name: "single host",
		source: &pb.Source_Walle{
			Project:  "",
			Tags:     []string{"rtc.reboot_segment-prestable-def"},
			Location: "",
		},
		prepare: func(w *walle.Fake) {
			w.On("GetHosts", mock.Anything).Return(&walle.GetHostsResponse{
				Hosts:      []*pb.Host{prestableReadyHost},
				Total:      1,
				NextCursor: 0,
			}, nil).Once()
			w.On("GetProject", "rtc-prestable").Return(prestableProject, nil).Once()
		},
		want:    []*pb.Host{prestableReadyHost},
		want1:   map[string]*pb.Project{"rtc-prestable": prestableProject},
		wantErr: false,
	}, {
		name: "empty source",
		source: &pb.Source_Walle{
			Project:  "",
			Tags:     []string{"rtc.reboot_segment-prestable-def"},
			Location: "",
		},
		prepare: func(w *walle.Fake) {
			w.On("GetHosts", mock.Anything).Return(&walle.GetHostsResponse{
				Hosts:      []*pb.Host{},
				Total:      0,
				NextCursor: 0,
			}, nil).Once()
		},
		want:    make([]*pb.Host, 0),
		want1:   make(map[string]*pb.Project),
		wantErr: false,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			l, _ := zap.New(zap.CLIConfig(log.DebugLevel))
			w := &walle.Fake{}
			tt.prepare(w)
			p := &Walle{source: tt.source}
			got, got1, err := p.Hosts(w, l)
			if (err != nil) != tt.wantErr {
				t.Errorf("Hosts() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Hosts() got = %v, want %v", got, tt.want)
			}
			if !reflect.DeepEqual(got1, tt.want1) {
				t.Errorf("Hosts() got1 = %v, want %v", got1, tt.want1)
			}
		})
	}
}
