package storages

import (
	"context"
	"fmt"
	"sync"

	"go.etcd.io/etcd/clientv3"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/maxwell/go/internal/storages/etcdutil"
	"a.yandex-team.ru/infra/maxwell/go/pkg/retry"
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

func NewYpNodesETCD(c *clientv3.Client) *YpNodesETCD {
	return &YpNodesETCD{c: c}
}

const (
	ypPrefix         = "ypnodes"
	nannyPrefix      = "nannySvc"
	ypSaveThreads    = 32
	nannySaveThreads = 32
)

type YpNodesETCD struct {
	c *clientv3.Client
}

type ypNodeSaver struct {
	results chan error
	sem     chan struct{}
	wg      sync.WaitGroup
	c       *clientv3.Client
}

func newSaver(threads int, c *clientv3.Client) *ypNodeSaver {
	return &ypNodeSaver{
		results: make(chan error, 4*threads),
		sem:     make(chan struct{}, threads),
		wg:      sync.WaitGroup{},
		c:       c,
	}
}

func (s *ypNodeSaver) lock() {
	s.wg.Add(1)
	s.sem <- struct{}{}
}

func (s *ypNodeSaver) unlock() {
	s.wg.Done()
	<-s.sem
}

func (s *ypNodeSaver) producer(nodes map[string]*pb.YpNode) {
	defer func() {
		s.wg.Wait()
		close(s.results)
	}()
	for _, v := range nodes {
		s.save(v)
	}
}

func (s *ypNodeSaver) save(n *pb.YpNode) {
	s.lock()
	go s.runSave(n)
}

func (s *ypNodeSaver) runSave(n *pb.YpNode) {
	key := fmt.Sprintf("/%s/%s", ypPrefix, n.MetaId)
	s.results <- retry.Retry(func() error {
		return etcdutil.PutProto(s.c, key, n)
	})
	s.unlock()
}

func (s *ypNodeSaver) collectErr() error {
	for r := range s.results {
		if r != nil {
			return r
		}
	}
	return nil
}

func (s *YpNodesETCD) PutNodes(nodes map[string]*pb.YpNode) error {
	ns := newSaver(ypSaveThreads, s.c)
	go ns.producer(nodes)
	return ns.collectErr()
}

func (s *YpNodesETCD) Pop(id string) error {
	key := fmt.Sprintf("/%s/%s", ypPrefix, id)
	_, err := s.c.Delete(context.Background(), key)
	return err
}

func (s *YpNodesETCD) Nodes() (map[string]*pb.YpNode, error) {
	key := fmt.Sprintf("/%s", ypPrefix)
	resp, err := s.c.Get(context.Background(), key, clientv3.WithPrefix())
	if err != nil {
		return nil, err
	}
	nodes := make(map[string]*pb.YpNode)
	for _, kv := range resp.Kvs {
		node := &pb.YpNode{}
		if err := proto.Unmarshal(kv.Value, node); err != nil {
			return nil, err
		}
		nodes[node.MetaId] = node
	}
	return nodes, nil
}

func (s *YpNodesETCD) Node(id string) (*pb.YpNode, error) {
	key := fmt.Sprintf("/%s/%s", ypPrefix, id)
	data, err := etcdutil.GetInterface(s.c, key)
	if err != nil {
		return nil, err
	}
	node := &pb.YpNode{}
	if err := proto.Unmarshal(data, node); err != nil {
		return nil, err
	}
	return node, nil
}

func NewNannyServicesETCD(c *clientv3.Client) *NannyServicesETCD {
	return &NannyServicesETCD{c: c}
}

type NannyServicesETCD struct {
	c *clientv3.Client
}

type nannySaver struct {
	results chan error
	sem     chan struct{}
	wg      sync.WaitGroup
	c       *clientv3.Client
}

func newNannySaver(threads int, c *clientv3.Client) *nannySaver {
	return &nannySaver{
		results: make(chan error, 4*threads),
		sem:     make(chan struct{}, threads),
		wg:      sync.WaitGroup{},
		c:       c,
	}
}

func (s *nannySaver) lock() {
	s.wg.Add(1)
	s.sem <- struct{}{}
}

func (s *nannySaver) unlock() {
	s.wg.Done()
	<-s.sem
}

func (s *nannySaver) producer(nodes map[string]*pb.NannyService) {
	defer func() {
		s.wg.Wait()
		close(s.results)
	}()
	for _, v := range nodes {
		s.save(v)
	}
}

func (s *nannySaver) save(n *pb.NannyService) {
	s.lock()
	go s.runSave(n)
}

func (s *nannySaver) runSave(n *pb.NannyService) {
	key := fmt.Sprintf("/%s/%s", ypPrefix, n.MetaId)
	s.results <- retry.Retry(func() error {
		return etcdutil.PutProto(s.c, key, n)
	})
	s.unlock()
}

func (s *nannySaver) collectErr() error {
	for r := range s.results {
		if r != nil {
			return r
		}
	}
	return nil
}

func (s *NannyServicesETCD) PutServices(services map[string]*pb.NannyService) error {
	ns := newNannySaver(nannySaveThreads, s.c)
	go ns.producer(services)
	return ns.collectErr()
}

func (s *NannyServicesETCD) Pop(id string) error {
	key := fmt.Sprintf("/%s/%s", nannyPrefix, id)
	_, err := s.c.Delete(context.Background(), key)
	return err
}

func (s *NannyServicesETCD) Services() (map[string]*pb.NannyService, error) {
	key := fmt.Sprintf("/%s", nannyPrefix)
	resp, err := s.c.Get(context.Background(), key, clientv3.WithPrefix())
	if err != nil {
		return nil, err
	}
	services := make(map[string]*pb.NannyService)
	for _, kv := range resp.Kvs {
		service := &pb.NannyService{}
		if err := proto.Unmarshal(kv.Value, service); err != nil {
			return nil, err
		}
		services[service.MetaId] = service
	}
	return services, nil
}

func (s *NannyServicesETCD) Service(id string) (*pb.NannyService, error) {
	key := fmt.Sprintf("/%s/%s", nannyPrefix, id)
	data, err := etcdutil.GetInterface(s.c, key)
	if err != nil {
		return nil, err
	}
	service := &pb.NannyService{}
	if err := proto.Unmarshal(data, service); err != nil {
		return nil, err
	}
	return service, nil
}
