package storages

import (
	"context"
	"fmt"

	"go.etcd.io/etcd/clientv3"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/maxwell/go/internal/storages/etcdutil"
	"a.yandex-team.ru/infra/maxwell/go/pkg/retry"
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

func NewHostsETCD(c *clientv3.Client) *HostsETCD {
	return &HostsETCD{c: c}
}

const hostsPrefix = "hosts"

type HostsETCD struct {
	c *clientv3.Client
}

func (s *HostsETCD) PutHosts(hosts map[string]*pb.Host) error {
	for _, host := range hosts {
		key := fmt.Sprintf("/%s/%s", hostsPrefix, host.Hostname)
		if err := retry.Retry(func() error {
			return etcdutil.PutProto(s.c, key, host)
		}); err != nil {
			return err
		}
	}
	return nil
}

func (s *HostsETCD) Pop(hostname string) error {
	key := fmt.Sprintf("/%s/%s", hostsPrefix, hostname)
	_, err := s.c.Delete(context.Background(), key)
	return err
}

func (s *HostsETCD) Hosts() (map[string]*pb.Host, error) {
	key := fmt.Sprintf("/%s", hostsPrefix)
	resp, err := s.c.Get(context.Background(), key, clientv3.WithPrefix())
	if err != nil {
		return nil, err
	}
	hosts := make(map[string]*pb.Host)
	for _, kv := range resp.Kvs {
		host := &pb.Host{}
		if err := proto.Unmarshal(kv.Value, host); err != nil {
			return nil, err
		}
		hosts[host.Hostname] = host
	}
	return hosts, nil
}

func (s *HostsETCD) Host(hostname string) (*pb.Host, error) {
	key := fmt.Sprintf("/%s/%s", hostsPrefix, hostname)
	data, err := etcdutil.GetInterface(s.c, key)
	if err != nil {
		return nil, err
	}
	host := &pb.Host{}
	if err := proto.Unmarshal(data, host); err != nil {
		return nil, err
	}
	return host, nil
}
