package storages

import (
	"context"
	"fmt"
	"strings"

	"go.etcd.io/etcd/clientv3"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/maxwell/go/internal/storages/etcdutil"
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

func NewJobsETCD(c *clientv3.Client) *JobsETCD {
	return &JobsETCD{c: c}
}

type JobsETCD struct {
	c *clientv3.Client
}

const jobsPrefix = "/jobs"

func (e *JobsETCD) Put(job *pb.Job) error {
	key := fmt.Sprintf("%s/%s", jobsPrefix, job.Spec.Name)
	return etcdutil.PutProto(e.c, key, job)
}

func (e *JobsETCD) Get(name string) (*pb.Job, error) {
	key := fmt.Sprintf("%s/%s", jobsPrefix, name)
	data, err := etcdutil.GetInterface(e.c, key)
	if err != nil {
		return nil, err
	}
	if data == nil {
		return nil, nil
	}
	j := &pb.Job{}
	if err := proto.Unmarshal(data, j); err != nil {
		return nil, err
	}
	return j, nil
}

func (e *JobsETCD) List() ([]*pb.Job, error) {
	resp, err := e.c.Get(context.Background(), jobsPrefix, clientv3.WithPrefix())
	if err != nil {
		return nil, err
	}
	jobs := make([]*pb.Job, len(resp.Kvs))
	for i, kv := range resp.Kvs {
		j := &pb.Job{}
		if err := proto.Unmarshal(kv.Value, j); err != nil {
			return nil, err
		}
		jobs[i] = j
	}
	return jobs, nil
}

func (e *JobsETCD) Names() ([]string, error) {
	resp, err := e.c.Get(context.Background(), jobsPrefix, clientv3.WithPrefix(), clientv3.WithKeysOnly())
	if err != nil {
		return nil, err
	}
	names := make([]string, len(resp.Kvs))
	for i, kv := range resp.Kvs {
		splitted := strings.Split(string(kv.Key), "/")
		hostname := splitted[len(splitted)-1]
		names[i] = hostname
	}
	return names, nil
}

func (e *JobsETCD) Delete(name string) error {
	key := fmt.Sprintf("%s/%s", jobsPrefix, name)
	_, err := e.c.Delete(context.Background(), key)
	return err
}
