package storages

import (
	"context"
	"encoding/json"
	"fmt"

	"go.etcd.io/etcd/clientv3"

	"a.yandex-team.ru/infra/maxwell/go/internal/storages/etcdutil"
)

func NewProcessedETCD(c *clientv3.Client, name string) *ProcessedETCD {
	return &ProcessedETCD{c: c, name: name}
}

type ProcessedETCD struct {
	c    *clientv3.Client
	name string
}

const (
	processedPrefix = "processed"
	enforcedPrefix  = "enforced"
)

func (e *ProcessedETCD) AppendProcessed(hostnames []string) error {
	processed, err := e.Processed()
	if err != nil {
		return err
	}
	newProcessed := append(processed, hostnames...)
	key := fmt.Sprintf("/%s/%s", e.name, processedPrefix)
	if err := etcdutil.PutInterface(e.c, key, newProcessed); err != nil {
		return err
	}
	return nil
}

func (e *ProcessedETCD) Processed() ([]string, error) {
	key := fmt.Sprintf("/%s/%s", e.name, processedPrefix)
	data, err := etcdutil.GetInterface(e.c, key)
	if err != nil {
		return nil, err
	}
	if data == nil {
		return make([]string, 0), nil
	}
	hostnames := make([]string, 0)
	if err := json.Unmarshal(data, &hostnames); err != nil {
		return nil, err
	}
	return hostnames, nil
}

func (e *ProcessedETCD) AppendEnforced(hostnames []string) error {
	enforced, err := e.Enforced()
	if err != nil {
		return err
	}
	newEnforced := append(enforced, hostnames...)
	key := fmt.Sprintf("/%s/%s", e.name, enforcedPrefix)
	if err := etcdutil.PutInterface(e.c, key, newEnforced); err != nil {
		return err
	}
	return nil
}

func (e *ProcessedETCD) Enforced() ([]string, error) {
	key := fmt.Sprintf("/%s/%s", e.name, enforcedPrefix)
	data, err := etcdutil.GetInterface(e.c, key)
	if err != nil {
		return nil, err
	}
	if data == nil {
		return make([]string, 0), nil
	}
	hostnames := make([]string, 0)
	if err := json.Unmarshal(data, &hostnames); err != nil {
		return nil, err
	}
	return hostnames, nil
}

func (e *ProcessedETCD) Delete() error {
	key := fmt.Sprintf("/%s/%s", e.name, processedPrefix)
	if _, err := e.c.Delete(context.Background(), key, clientv3.WithPrefix()); err != nil {
		return err
	}
	key = fmt.Sprintf("/%s/%s", e.name, enforcedPrefix)
	_, err := e.c.Delete(context.Background(), key, clientv3.WithPrefix())
	return err
}
