package storages

import (
	"go.etcd.io/etcd/clientv3"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

type Job interface {
	Tasks
	Pool
	Processed
	Delete() error
}

type Jobs interface {
	Put(job *pb.Job) error
	Get(name string) (*pb.Job, error)
	List() ([]*pb.Job, error)
	Names() ([]string, error)
	Delete(name string) error
}

type Manager interface {
	Master() (string, error)
}

type Tasks interface {
	PutTasks([]*pb.Task) error
	Tasks() ([]*pb.Task, error)
}

// Processed storage may contains duplicates.
// Duplicates it is ok for mode=follow jobs
type Processed interface {
	AppendProcessed([]string) error
	Processed() ([]string, error)
	AppendEnforced([]string) error
	Enforced() ([]string, error)
}

type Pool interface {
	PutGroups(map[string][]string) error
	Groups() (map[string][]string, error)
	PutGroupsOrder([]string) error
	GroupsOrder() ([]string, error)
}

type Hosts interface {
	PutHosts(map[string]*pb.Host) error
	Hosts() (map[string]*pb.Host, error)
	Pop(string) error
	Host(hostname string) (*pb.Host, error)
}

type YpNodes interface {
	PutNodes(map[string]*pb.YpNode) error
	Nodes() (map[string]*pb.YpNode, error)
	Pop(string) error
	Node(id string) (*pb.YpNode, error)
}

type NannyServices interface {
	PutServices(map[string]*pb.NannyService) error
	Services() (map[string]*pb.NannyService, error)
	Pop(string) error
	Service(id string) (*pb.NannyService, error)
}

type Projects interface {
	PutProjects(map[string]*pb.Project) error
	Projects() (map[string]*pb.Project, error)
}

func NewJobs(inMemory bool, c *clientv3.Client) Jobs {
	if inMemory {
		return NewJobsInMemory()
	} else {
		return NewJobsETCD(c)
	}
}

func NewManager(inMemory bool, c *clientv3.Client) Manager {
	if inMemory {
		return NewManagerInMemory()
	} else {
		return NewManagerETCD(c)
	}
}

func NewJob(inMemory bool, c *clientv3.Client, name string) Job {
	if inMemory {
		return NewJobInMemory()
	} else {
		return NewJobETCD(c, name)
	}
}

func NewHosts(inMemory bool, c *clientv3.Client) Hosts {
	if inMemory {
		return NewHostsInMemory()
	} else {
		return NewHostsETCD(c)
	}
}

func NewYpNodes(inMemory bool, c *clientv3.Client) YpNodes {
	if inMemory {
		return NewYpNodesInMemory()
	} else {
		return NewYpNodesETCD(c)
	}
}

func NewNannyServices(inMemory bool, c *clientv3.Client) NannyServices {
	if inMemory {
		return NewNannyServicesInMemory()
	} else {
		return NewNannyServicesETCD(c)
	}

}

func NewJobInMemory() *JobInMemory {
	return &JobInMemory{
		TasksInMemory:     NewTasksInMemory(),
		PoolInMemory:      NewPoolInMemory(),
		ProcessedInMemory: NewProcessedInMemory(),
	}
}

func NewJobETCD(c *clientv3.Client, name string) *JobETCD {
	return &JobETCD{
		TasksETCD:     NewTasksETCD(c, name),
		PoolETCD:      NewPoolETCD(c, name),
		ProcessedETCD: NewProcessedETCD(c, name),
	}
}

type JobInMemory struct {
	*TasksInMemory
	*PoolInMemory
	*ProcessedInMemory
}

func (s *JobInMemory) Delete() error {
	return nil
}

type JobETCD struct {
	*TasksETCD
	*PoolETCD
	*ProcessedETCD
}

func (s *JobETCD) Delete() error {
	if err := s.TasksETCD.Delete(); err != nil {
		return err
	}
	if err := s.PoolETCD.Delete(); err != nil {
		return err
	}
	if err := s.ProcessedETCD.Delete(); err != nil {
		return err
	}
	return nil
}
