package storages

import (
	"context"
	"fmt"

	"go.etcd.io/etcd/clientv3"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/infra/maxwell/go/internal/storages/etcdutil"
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

func NewTasksETCD(c *clientv3.Client, name string) *TasksETCD {
	return &TasksETCD{c: c, name: name}
}

type TasksETCD struct {
	c    *clientv3.Client
	name string
}

const wsPrefix = "working-set"

func (e *TasksETCD) PutTasks(records []*pb.Task) error {
	old, err := e.Tasks()
	if err != nil {
		return err
	}
	// Overwriting current records
	for i, task := range records {
		key := fmt.Sprintf("/%s/%s/%d", e.name, wsPrefix, i)
		if err := etcdutil.PutProto(e.c, key, task); err != nil {
			return err
		}
	}
	// Trimming tail of working set
	for i := len(records); i < len(old); i++ {
		key := fmt.Sprintf("/%s/%s/%d", e.name, wsPrefix, i)
		ctx := context.Background()
		if _, err := e.c.Delete(ctx, key); err != nil {
			return err
		}
	}
	return nil
}

func (e *TasksETCD) Tasks() ([]*pb.Task, error) {
	key := fmt.Sprintf("/%s/%s", e.name, wsPrefix)
	resp, err := e.c.Get(context.Background(), key, clientv3.WithPrefix())
	if err != nil {
		return nil, err
	}
	records := make([]*pb.Task, len(resp.Kvs))
	for i, kv := range resp.Kvs {
		r := &pb.Task{}
		if err := proto.Unmarshal(kv.Value, r); err != nil {
			return nil, err
		}
		records[i] = r
	}
	return records, nil
}

func (e *TasksETCD) Delete() error {
	key := fmt.Sprintf("/%s/%s", e.name, wsPrefix)
	_, err := e.c.Delete(context.Background(), key, clientv3.WithPrefix())
	return err
}
