package sync

import (
	"context"
	"io/ioutil"
	"os"
	"path"
	"time"

	"a.yandex-team.ru/infra/maxwell/go/internal/job"
	"a.yandex-team.ru/infra/maxwell/go/internal/tick"
	"a.yandex-team.ru/infra/maxwell/go/pkg/yamlutils"
	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

type Manager struct {
	jobm         *job.Manager
	specsDirName string
	l            *zap.Logger
	name         string
}

func (m *Manager) Start() error {
	p := tick.NewPeriodic(m.name, 2*time.Minute, m)
	ctx := context.Background()
	m.l.Info("Starting periodic " + m.name)
	m.l.Info("Syncing specs in: " + m.specsDirName)
	return p.Run(ctx)
}

func CreateManager(p string, manager *job.Manager) *Manager {
	l, _ := zap.New(zap.ConsoleConfig(log.InfoLevel))
	wd, _ := os.Getwd()
	return &Manager{
		name:         "specman",
		jobm:         manager,
		specsDirName: path.Join(wd, p),
		l:            l,
	}
}

func (m *Manager) Sync(ctx context.Context) error {
	files, err := ioutil.ReadDir(m.specsDirName)
	if err != nil {
		return err
	}
	specsCollection := make([]*pb.Job_Spec, 0)
	for _, f := range files {
		specFilename := path.Join(m.specsDirName, f.Name())
		s := &pb.Job_Spec{}
		err := yamlutils.ReadFromFile(specFilename, s)
		if err != nil {
			m.l.Error("can not read spec", log.String("filename", specFilename), log.String("error", err.Error()))
			continue
		}
		specsCollection = append(specsCollection, s)
	}
	for _, spec := range specsCollection {
		err = m.jobm.PutJob(spec, "specman")
		if err != nil {
			return err
		}
	}
	return nil
}
