package tasks

import (
	"math"
	"strings"
	"time"

	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/duration"

	"a.yandex-team.ru/infra/maxwell/go/proto"
)

func checkHostHealthChecks(status string, checks map[string]string, finishChecks []string) bool {
	for _, s := range finishChecks {
		if checks[s] != "passed" {
			return false
		}
	}
	return status == "ready"
}

func UpdateState(task *pb.Task, host *pb.Host, spec *pb.Job_Spec) {
	updateState(task, host, spec)
	updateEnforce(task, host, spec)
	// Update mtime of task update
	task.Meta.Mtime = ptypes.TimestampNow()
}

func updateState(task *pb.Task, host *pb.Host, spec *pb.Job_Spec) {
	// check host satisfy finish checks
	if host.Status == "ready" && checkHostHealthChecks(host.Status, host.Health, spec.PostCondition.Health.Check) {
		task.State = pb.Task_FINISHED
		return
	}
	switch task.State {
	case pb.Task_CREATED:
		task.State = updateCreatedTask(host)
	case pb.Task_RUNNING:
		task.State = updateRunningTask(makeTimeouts(spec.Timeouts), task.Meta, host)
	}
}

func updateEnforce(task *pb.Task, host *pb.Host, spec *pb.Job_Spec) {
	switch task.Enforce {
	case pb.Task_GRACEFUL:
		task.Enforce = checkEnforceGraceful(makeTimeouts(spec.Timeouts), task.Meta, host)
	case pb.Task_CANCEL_WAITING:
		task.Enforce = checkEnforceCancelWaiting(makeTimeouts(spec.Timeouts), task.Meta, host)
	}
}

func checkEnforceGraceful(timeouts *pb.Job_Spec_Timeouts, meta *pb.Task_Meta, host *pb.Host) pb.Task_Enforce {
	notModified := func() pb.Task_Enforce {
		return pb.Task_GRACEFUL
	}
	// Enforce disabled
	if timeouts.Enforce == nil {
		return notModified()
	}
	creationTime, err := ptypes.Timestamp(meta.CreationTime)
	if err != nil {
		return notModified()
	}
	if host.LastTask == nil {
		return notModified()
	}
	lastTaskTS, err := ptypes.Timestamp(host.LastTask.Time)
	if err != nil {
		return notModified()
	}
	// delta between Maxwell/wall-e task creation
	delta := math.Abs(float64(lastTaskTS.Sub(creationTime)))
	// eps: assume that was not changed in first 2 minutes after creation
	eps := float64(2 * time.Minute)
	if time.Since(creationTime) > dur(timeouts.Enforce) &&
		// Enforce needed only when we acquiring cms permission too long
		host.Task == "acquire-permission:cms" &&
		// Check that wall-e task was not changed since maxwell task creation
		delta < eps &&
		// Additional check to be more safe
		strings.HasPrefix(host.LastTask.Description, "Maxwell:") {
		return pb.Task_NEED_CANCEL
	}
	return notModified()
}

func checkEnforceCancelWaiting(timeouts *pb.Job_Spec_Timeouts, meta *pb.Task_Meta, host *pb.Host) pb.Task_Enforce {
	// Check that host became free after task canceling
	if host.Status == "ready" {
		return pb.Task_NEED_ENFORCE
	}
	return pb.Task_CANCEL_WAITING
}

func updateRunningTask(timeouts *pb.Job_Spec_Timeouts, meta *pb.Task_Meta, host *pb.Host) pb.Task_State {
	notModified := func() pb.Task_State {
		return pb.Task_RUNNING
	}
	creationTime, err := ptypes.Timestamp(meta.CreationTime)
	if err != nil {
		panic(err)
	}
	// Do not keep deed hosts in RUNNING state
	if host.Status == "dead" {
		return pb.Task_WAITING
	}
	if strings.HasPrefix(strings.ToLower(host.Ticket), "itdc") {
		if time.Since(creationTime) > dur(timeouts.Itdc) {
			return pb.Task_WAITING
		}
		return notModified()
	}
	if host.Task == "acquire-permission:cms" {
		if time.Since(creationTime) > dur(timeouts.Cms) {
			return pb.Task_WAITING
		}
		return notModified()
	}
	if time.Since(creationTime) > dur(timeouts.Default) {
		return pb.Task_WAITING
	}
	return notModified()
}

func updateCreatedTask(host *pb.Host) pb.Task_State {
	if host.Status != "ready" {
		return pb.Task_RUNNING
	}
	return pb.Task_CREATED
}

func makeTimeouts(overrides *pb.Job_Spec_Timeouts) *pb.Job_Spec_Timeouts {
	// Create timeouts here to hide vars from outside
	const (
		CMSTimeout     = 6 * time.Hour
		ITDCTimeout    = 6 * time.Hour
		DefaultTimeout = 3 * time.Hour
	)
	rv := &pb.Job_Spec_Timeouts{
		Cms:     ptypes.DurationProto(CMSTimeout),
		Itdc:    ptypes.DurationProto(ITDCTimeout),
		Default: ptypes.DurationProto(DefaultTimeout),
	}
	// No overrides
	if overrides == nil {
		return rv
	}
	if overrides.Cms != nil {
		rv.Cms = overrides.Cms
	}
	if overrides.Default != nil {
		rv.Default = overrides.Default
	}
	if overrides.Itdc != nil {
		rv.Itdc = overrides.Itdc
	}
	if overrides.Enforce != nil {
		rv.Enforce = overrides.Enforce
	}
	return rv
}

func dur(p *duration.Duration) time.Duration {
	// Ignore errors. We validating spec and our timeouts values are good.
	d, err := ptypes.Duration(p)
	if err != nil {
		panic(err)
	}
	return d
}
