package tasks

import (
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/durationpb"
	"google.golang.org/protobuf/types/known/timestamppb"
	"testing"
	"time"

	"github.com/golang/protobuf/ptypes"

	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/test/assertpb"
)

var (
	readyHost = &pb.Host{
		Status: "ready",
		Health: map[string]string{"hostman-ready": "passed", "check_iss_agent": "passed", "hostman-distrib": "passed"},
	}
	acquireCMSHost = &pb.Host{
		Status: "deploying",
		Health: map[string]string{"hostman-ready": "passed", "check_iss_agent": "passed"},
		Task:   "acquire-permission:cms",
	}
	deployingHost = &pb.Host{
		Status: "deploying",
		Health: make(map[string]string),
		Task:   "monitor",
	}
	finishHealthChecks = []string{"hostman-ready", "check_iss_agent", "hostman-distrib"}
)

func TestUpdateState(t *testing.T) {
	tests := []struct {
		before       pb.Task_State
		after        pb.Task_State
		name         string
		creationTime time.Time
		host         *pb.Host
		checkHealth  []string
	}{{
		name:         "finished",
		creationTime: time.Now(),
		before:       pb.Task_RUNNING,
		after:        pb.Task_FINISHED,
		host:         readyHost,
		checkHealth:  finishHealthChecks,
	}, {
		name:         "waiting on cms timeout",
		creationTime: time.Now().Add(-7 * time.Hour),
		before:       pb.Task_RUNNING,
		after:        pb.Task_WAITING,
		host:         acquireCMSHost,
		checkHealth:  finishHealthChecks,
	}, {
		name:         "running before cms timeout",
		creationTime: time.Now().Add(-4 * time.Hour),
		before:       pb.Task_RUNNING,
		after:        pb.Task_RUNNING,
		host:         acquireCMSHost,
		checkHealth:  finishHealthChecks,
	}, {
		name:         "waiting on run timeout",
		creationTime: time.Now().Add(-4 * time.Hour),
		before:       pb.Task_RUNNING,
		after:        pb.Task_WAITING,
		host:         deployingHost,
		checkHealth:  finishHealthChecks,
	}, {
		name:         "running before run timeout",
		creationTime: time.Now().Add(-2 * time.Hour),
		before:       pb.Task_RUNNING,
		after:        pb.Task_RUNNING,
		host:         deployingHost,
		checkHealth:  finishHealthChecks,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			ts, _ := ptypes.TimestampProto(tt.creationTime)
			task := &pb.Task{
				Meta: &pb.Task_Meta{
					Action:       "reboot",
					CreationTime: ts,
					Mtime:        ts,
				},
				State:    tt.before,
				Hostname: "hostname",
				Group:    "test_group",
			}
			spec := &pb.Job_Spec{
				PostCondition: &pb.PostCondition{
					Health: &pb.PostCondition_Health{Check: tt.checkHealth},
				},
			}
			UpdateState(task, tt.host, spec)
			assertpb.Equal(t, tt.after, task.State)
		})
	}
}

func Test_updateEnforce(t *testing.T) {
	now := time.Now()
	tests := []struct {
		name           string
		creationTime   time.Time
		enforceTimeout *durationpb.Duration
		before         pb.Task_Enforce
		after          pb.Task_Enforce
		host           func(h *pb.Host)
	}{{
		name:           "graceful",
		creationTime:   now,
		enforceTimeout: nil, // enforce disabled
		before:         pb.Task_GRACEFUL,
		after:          pb.Task_GRACEFUL,
		host:           func(h *pb.Host) {},
	}, {
		name:           "graceful -> need cancel",
		creationTime:   now.Add(-2 * time.Hour),
		enforceTimeout: durationpb.New(time.Hour),
		before:         pb.Task_GRACEFUL,
		after:          pb.Task_NEED_CANCEL,
		host: func(h *pb.Host) {
			h.LastTask = &pb.Host_Task{
				Time:        timestamppb.New(now.Add(-2 * time.Hour)),
				Description: "Maxwell: do smth",
			}
		},
	}, {
		name:           "graceful -> graceful (different tasks creation time walle/maxwell)",
		creationTime:   now.Add(-2 * time.Hour),
		enforceTimeout: durationpb.New(time.Hour),
		before:         pb.Task_GRACEFUL,
		after:          pb.Task_GRACEFUL,
		host: func(h *pb.Host) {
			h.LastTask = &pb.Host_Task{
				Time:        timestamppb.New(now.Add(-time.Hour)),
				Description: "Maxwell: do smth",
			}
		},
	}, {
		name:           "graceful -> graceful (not maxwell task description)",
		creationTime:   now.Add(-2 * time.Hour),
		enforceTimeout: durationpb.New(time.Hour),
		before:         pb.Task_GRACEFUL,
		after:          pb.Task_GRACEFUL,
		host: func(h *pb.Host) {
			h.LastTask = &pb.Host_Task{
				Time:        timestamppb.New(now.Add(-2 * time.Hour)),
				Description: "Disk check failed. hw-watcher:",
			}
		},
	}, {
		name:           "graceful -> graceful (enforce timeout not exceeded)",
		creationTime:   now.Add(-2 * time.Hour),
		enforceTimeout: durationpb.New(3 * time.Hour),
		before:         pb.Task_GRACEFUL,
		after:          pb.Task_GRACEFUL,
		host: func(h *pb.Host) {
			h.LastTask = &pb.Host_Task{
				Time:        timestamppb.New(now.Add(-2 * time.Hour)),
				Description: "Maxwell: do smth",
			}
		},
	}, {
		name:           "cancel_waiting -> cancel_waiting",
		creationTime:   now,
		enforceTimeout: durationpb.New(1 * time.Hour),
		before:         pb.Task_CANCEL_WAITING,
		after:          pb.Task_CANCEL_WAITING,
		host:           func(h *pb.Host) {},
	}, {
		name:           "cancel_waiting -> need_enforce",
		creationTime:   now,
		enforceTimeout: durationpb.New(1 * time.Hour),
		before:         pb.Task_CANCEL_WAITING,
		after:          pb.Task_NEED_ENFORCE,
		host: func(h *pb.Host) {
			h.Status = "ready"
		},
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			ts, _ := ptypes.TimestampProto(tt.creationTime)
			task := &pb.Task{
				Meta: &pb.Task_Meta{
					Action:       "reboot",
					CreationTime: ts,
					Mtime:        ts,
				},
				Enforce:  tt.before,
				Hostname: "hostname",
				Group:    "test_group",
			}
			spec := &pb.Job_Spec{
				PostCondition: &pb.PostCondition{
					Health: &pb.PostCondition_Health{Check: make([]string, 0)},
				},
				Timeouts: makeTimeouts(&pb.Job_Spec_Timeouts{
					Enforce: tt.enforceTimeout,
				}),
			}
			h := (proto.Clone(acquireCMSHost)).(*pb.Host)
			tt.host(h)
			updateEnforce(task, h, spec)
			assertpb.Equal(t, tt.after, task.Enforce)
		})
	}
}
