package tasks

import (
	"fmt"
	"time"

	"github.com/golang/protobuf/ptypes"

	"a.yandex-team.ru/infra/maxwell/go/pkg/walle"
	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
)

// walle_firmware types valid for send host to profile with "drives_update"
var checksToSendToDrivesUpdate = []string{"SSD", "HDD", "NVME"}

type Task interface {
	Start(w walle.IClient)
	UpdateStatus(w walle.IClient) Task
	Proto() *pb.Task
}

func CreateTask(taskName, group string, host *pb.Host) *pb.Task {
	if slices.ContainsString([]string{AutoAction.Type(), AutoFirmwareUpdateAction.Type(), AutoRedeployAction.Type()}, taskName) {
		taskName = TaskTypeFromAuto(taskName, host)
	}
	return &pb.Task{
		Meta: &pb.Task_Meta{
			Action:       taskName,
			CreationTime: ptypes.TimestampNow(),
			Mtime:        ptypes.TimestampNow(),
		},
		State:    pb.Task_CREATED,
		Hostname: host.Hostname,
		Group:    group,
	}
}

func StartTask(p *pb.Task, spec *pb.Job_Spec, l log.Logger, w walle.IClient, dryRun, enforce bool) error {
	var t *WalleTask
	reason := fmt.Sprintf("Maxwell: %s '%s'", p.Meta.Action, spec.Name)
	ignoreCMS := ignoreCMSFromSafetyLevel(spec.SafetyLevel) || enforce
	switch p.Meta.Action {
	case RedeployAction.Type():
		t = Redeploy(reason, ignoreCMS)
	case FirmwareUpdateWithRedeployAction.Type():
		t = FirmwareUpdateWithRedeploy(reason, ignoreCMS)
	case DrivesUpdateWithRedeployAction.Type():
		t = DrivesUpdateWithRedeploy(reason, ignoreCMS)
	case FirmwareUpdateAction.Type():
		t = FirmwareUpdate(reason, ignoreCMS)
	case DrivesUpdateAction.Type():
		t = DrivesUpdate(reason, ignoreCMS)
	case RebootAction.Type():
		t = Reboot(reason, ignoreCMS, spec.Ssh, spec.Kexec)
	case FullProfileWithRedeployAction.Type():
		t = FullProfileWithRedeploy(reason, ignoreCMS)
	case FullProfileAction.Type():
		t = FullProfile(reason, ignoreCMS)
	case WalleHandleFailureAction.Type():
		t = WalleHandleFailure(reason, ignoreCMS)
	default:
		return fmt.Errorf("unknown task type: %s", p.Meta.Action)
	}
	if dryRun {
		if err := t.DryStart(l, p.Hostname); err != nil {
			l.Errorf("Failed to start task: %s", err)
		}
	} else {
		if err := t.Start(l, w, p.Hostname); err != nil {
			l.Errorf("Failed to start task: %s", err)
		}
	}
	return nil
}

type WalleChecks struct {
	hostmanDistrib   *walle.HealthCheck
	walleFirmware    *walle.HealthCheck
	needRebootKernel *walle.HealthCheck
}

type taskActions struct {
	reboot             bool
	redeploy           bool
	drivesUpdate       bool
	fullFirmwareUpdate bool
}

func TaskTypeFromAuto(autoTaskType string, host *pb.Host) string {
	neededActions := GetActionsByWalleChecks(host)
	allowedActions := AutoTaskToAllowedActions(autoTaskType)
	return TaskNameByActions(&taskActions{
		reboot:             neededActions.reboot && allowedActions.reboot,
		redeploy:           neededActions.redeploy && allowedActions.redeploy,
		drivesUpdate:       neededActions.drivesUpdate && allowedActions.drivesUpdate,
		fullFirmwareUpdate: neededActions.fullFirmwareUpdate && allowedActions.fullFirmwareUpdate,
	})
}

func GetWalleHealthChecks(hostname string, w walle.IClient) (*WalleChecks, error) {
	onFailSleepDuration := 30 * time.Second
	const retiesLimit = 5
	var err error
	for i := 0; i < retiesLimit; i++ {
		hostmanDistrib, err := w.GetHealthCheck(hostname, "hostman-distrib")
		if err != nil {
			time.Sleep(onFailSleepDuration)
			continue
		}
		walleFirmware, err := w.GetHealthCheck(hostname, "walle_firmware")
		if err != nil {
			time.Sleep(onFailSleepDuration)
			continue
		}
		needRebootKernel, err := w.GetHealthCheck(hostname, "need_reboot_kernel")
		if err != nil {
			time.Sleep(onFailSleepDuration)
			continue
		}
		return &WalleChecks{
			hostmanDistrib:   hostmanDistrib,
			walleFirmware:    walleFirmware,
			needRebootKernel: needRebootKernel,
		}, nil
	}
	return nil, err
}

const (
	hostmanDistribCheck   = "hostman-distrib"
	needRebootKernelCheck = "need_reboot_kernel"
	walleFirmwareCheck    = "walle_firmware"
)

func GetActionsByWalleChecks(host *pb.Host) *taskActions {
	health := host.Health
	if health[walleFirmwareCheck] == "failed" {
		// if walle_firmware check metadata contains only from ["SSD", "HDD", "NVME"]
		// we can send host in profile with drives_update tag
		if containsAll(checksToSendToDrivesUpdate, host.FirmwareProblems) {
			if health[hostmanDistribCheck] == "failed" {
				return &taskActions{redeploy: true, drivesUpdate: true}
			}
			return &taskActions{drivesUpdate: true}
		} else {
			if health[hostmanDistribCheck] == "failed" {
				return &taskActions{redeploy: true, fullFirmwareUpdate: true}
			}
			return &taskActions{fullFirmwareUpdate: true}
		}
	}
	if health[hostmanDistribCheck] == "failed" {
		return &taskActions{redeploy: true}
	}
	if health[needRebootKernelCheck] == "failed" {
		return &taskActions{reboot: true}
	}
	return &taskActions{}
}

// Actions to tasks matching
//
// redeploy + drives_update    -> drives_update with redeploy tag
// drives_update               -> drives_update
// redeploy + firmware_update  -> firmware_update with redeploy tag
// firmware_update             -> firmware_update
// redeploy                    -> redeploy
// reboot                      -> redeploy
func TaskNameByActions(actions *taskActions) string {
	if actions.fullFirmwareUpdate {
		if actions.redeploy {
			return FirmwareUpdateWithRedeployAction.Type()
		}
		return FirmwareUpdateAction.Type()
	}
	if actions.drivesUpdate {
		if actions.redeploy {
			return DrivesUpdateWithRedeployAction.Type()
		}
		return DrivesUpdateAction.Type()
	}
	if actions.redeploy {
		return RedeployAction.Type()
	}
	if actions.reboot {
		return RebootAction.Type()
	}
	return UnknownAction.Type()
}

// AutoTasks to allowed actions matching
//
// auto                 -> redeploy, reboot, firmware_update, drives_update
// auto_redeploy        -> redeploy, firmware_update, drives_update
// auto_firmware_update -> firmware_update, drives_update
func AutoTaskToAllowedActions(taskType string) *taskActions {
	switch taskType {
	case AutoAction.Type():
		return &taskActions{
			reboot:             true,
			redeploy:           true,
			drivesUpdate:       true,
			fullFirmwareUpdate: true,
		}
	case AutoRedeployAction.Type():
		return &taskActions{
			redeploy:           true,
			drivesUpdate:       true,
			fullFirmwareUpdate: true,
		}
	case AutoFirmwareUpdateAction.Type():
		return &taskActions{
			drivesUpdate:       true,
			fullFirmwareUpdate: true,
		}
	}
	return &taskActions{}
}

func containsAll(haystack []string, needle []string) bool {
	for _, e := range needle {
		if !slices.ContainsString(haystack, e) {
			return false
		}
	}
	return true
}
