package tasks

import (
	"testing"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

func TestTaskTypeFromAuto(t *testing.T) {
	tests := []struct {
		name             string
		action           string
		health           map[string]string
		firmwareProblems []string
		want             string
	}{{
		name:   "auto_redeploy task (hostman_distrib && walle_firmware(['SSD'] && need_reboot_kernel))",
		action: AutoRedeployAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "failed",
		},
		firmwareProblems: []string{"SSD"},
		want:             DrivesUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto_redeploy task (hostman_distrib && walle_firmware(['SSD']))",
		action: AutoRedeployAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"SSD"},
		want:             DrivesUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto_redeploy task (hostman_distrib && walle_firmware(['BIOS']))",
		action: AutoRedeployAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"BIOS"},
		want:             FirmwareUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto_redeploy task (hostman_distrib && walle_firmware(['BIOS'] && need_reboot_kernel))",
		action: AutoRedeployAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "failed",
		},
		firmwareProblems: []string{"BIOS"},
		want:             FirmwareUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto_redeploy task (hostman_distrib && walle_firmware(['SSD', 'BIOS'] && need_reboot_kernel))",
		action: AutoRedeployAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "failed",
		},
		firmwareProblems: []string{"SSD", "BIOS"},
		want:             FirmwareUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto_redeploy task (walle_firmware(['BIOS']))",
		action: AutoRedeployAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "passed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"SSD"},
		want:             DrivesUpdateAction.Type(),
	}, {
		name:   "auto_redeploy task (walle_firmware(['SSD']))",
		action: AutoRedeployAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "passed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"SSD"},
		want:             DrivesUpdateAction.Type(),
	}, {
		name:   "auto_redeploy task (hostman-distrib)",
		action: AutoRedeployAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "passed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{},
		want:             RedeployAction.Type(),
	}, {
		name:   "auto_firmware_update task (walle_firmware(['SSD']))",
		action: AutoFirmwareUpdateAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "passed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"SSD"},
		want:             DrivesUpdateAction.Type(),
	}, {
		name:   "auto_firmware_update task (walle_firmware(['BIOS']))",
		action: AutoFirmwareUpdateAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "passed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"BIOS"},
		want:             FirmwareUpdateAction.Type(),
	}, {
		name:   "auto_firmware_update task (hostman-distrib, walle_firmware(['BIOS']))",
		action: AutoFirmwareUpdateAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"BIOS"},
		want:             FirmwareUpdateAction.Type(),
	}, {
		name:   "auto_firmware_update task (hostman-distrib)",
		action: AutoFirmwareUpdateAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "passed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{},
		want:             UnknownAction.Type(),
	}, {
		name:   "auto task (hostman-distrib, walle_firmware(['BIOS']))",
		action: AutoRedeployAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"BIOS"},
		want:             FirmwareUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto task (hostman-distrib, walle_firmware(['SSD']))",
		action: AutoAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"SSD"},
		want:             DrivesUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto task (hostman-distrib, walle_firmware(['SSD'], need_reboot_kernel))",
		action: AutoAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "failed",
		},
		firmwareProblems: []string{"SSD"},
		want:             DrivesUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto task (hostman-distrib, walle_firmware(['BIOS'], need_reboot_kernel))",
		action: AutoAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "failed",
		},
		firmwareProblems: []string{"BIOS"},
		want:             FirmwareUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto task (hostman-distrib, walle_firmware(['BIOS']))",
		action: AutoAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"BIOS"},
		want:             FirmwareUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto task (hostman-distrib, walle_firmware(['SSD']))",
		action: AutoAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "failed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{"SSD"},
		want:             DrivesUpdateWithRedeployAction.Type(),
	}, {
		name:   "auto task (hostman-distrib, need_reboot_kernel)",
		action: AutoAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "failed",
			"walle_firmware":     "passed",
			"need_reboot_kernel": "failed",
		},
		firmwareProblems: []string{},
		want:             RedeployAction.Type(),
	}, {
		name:   "auto task (need_reboot_kernel)",
		action: AutoAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "passed",
			"walle_firmware":     "passed",
			"need_reboot_kernel": "failed",
		},
		firmwareProblems: []string{},
		want:             RebootAction.Type(),
	}, {
		name:   "auto task ()",
		action: AutoAction.Type(),
		health: map[string]string{
			"hostman-distrib":    "passed",
			"walle_firmware":     "passed",
			"need_reboot_kernel": "passed",
		},
		firmwareProblems: []string{},
		want:             UnknownAction.Type(),
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			host := &pb.Host{
				Health:           tt.health,
				FirmwareProblems: tt.firmwareProblems,
			}
			if got := TaskTypeFromAuto(tt.action, host); got != tt.want {
				t.Errorf("TaskTypeFromAuto() = %v, want %v", got, tt.want)
			}
		})
	}
}
