package tasks

import (
	"a.yandex-team.ru/infra/maxwell/go/pkg/retry"
	"a.yandex-team.ru/infra/maxwell/go/pkg/walle"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
)

type WalleTask struct {
	Description string
	Action      WalleAction
}

func (t *WalleTask) Start(l log.Logger, w walle.IClient, hostname string) error {
	return retry.Retry(func() error {
		l.Infof("Starting '%s' on '%s'", t.Description, hostname)
		err := t.Action(w, hostname)
		if err != nil {
			l.Errorf("Failed to start task: %s", err)
			started, checkStartedErr := CheckStarted(l, w, hostname)
			if !started || checkStartedErr != nil {
				return err
			}
		}
		return nil
	})
}

// DryStart only print logs
func (t *WalleTask) DryStart(l log.Logger, hostname string) error {
	return retry.Retry(func() error {
		l.Infof("[dry-run] Starting '%s' on '%s'", t.Description, hostname)
		return nil
	})
}

type WalleAction func(w walle.IClient, hostname string) error

// safety levels
const (
	SafetyLevelNormal    = "NORMAL"
	SafetyLevelIgnoreCMS = "IGNORE_CMS"
	SafetyLevelIgnoreAll = "IGNORE_ALL"
)

// sshAllowedMethods
const (
	SSHForbid   = "forbid"
	SSHOnly     = "only"
	SSHFallback = "fallback"
)

// tasksIDs
var (
	RedeployAction                   = NewTaskType("redeploy", []string{"automated-reboot", "automated-redeploy", "redeploy", "reboot", "automation"})
	AutoAction                       = NewTaskType("auto", []string{"automated-reboot", "automated-redeploy", "automated-profile", "redeploy", "profile", "reboot", "automation"})
	AutoRedeployAction               = NewTaskType("auto_redeploy", []string{"automated-reboot", "automated-redeploy", "automated-profile", "redeploy", "profile", "reboot", "automation"})
	AutoFirmwareUpdateAction         = NewTaskType("auto_firmware_update", []string{"automated-reboot", "automated-redeploy", "automated-profile", "redeploy", "profile", "reboot", "automation"})
	DrivesUpdateAction               = NewTaskType("drives_update", []string{"automated-reboot", "automated-profile", "profile", "reboot", "automation"})
	DrivesUpdateWithRedeployAction   = NewTaskType("drives_update_with_redeploy", []string{"automated-reboot", "automated-redeploy", "automated-profile", "redeploy", "profile", "reboot", "automation"})
	FirmwareUpdateAction             = NewTaskType("firmware_update", []string{"automated-reboot", "automated-profile", "profile", "reboot", "automation"})
	FirmwareUpdateWithRedeployAction = NewTaskType("firmware_update_with_redeploy", []string{"automated-reboot", "automated-redeploy", "automated-profile", "redeploy", "profile", "reboot", "automation"})
	FullProfileAction                = NewTaskType("full_profiling", []string{"automated-reboot", "automated-profile", "profile", "reboot", "automation"})
	FullProfileWithRedeployAction    = NewTaskType("full_profiling_with_redeploy", []string{"automated-reboot", "automated-redeploy", "automated-profile", "redeploy", "profile", "reboot", "automation"})
	RebootAction                     = NewTaskType("reboot", []string{"automated-reboot", "reboot", "automation"})
	WalleHandleFailureAction         = NewTaskType("walle_handle_failure", []string{"automation"})
	UnknownAction                    = NewTaskType("unknown", make([]string, 0))
)

var ValidActions = []*TaskType{
	AutoAction,
	RedeployAction,
	AutoRedeployAction,
	AutoFirmwareUpdateAction,
	DrivesUpdateAction,
	DrivesUpdateWithRedeployAction,
	FirmwareUpdateAction,
	FirmwareUpdateWithRedeployAction,
	FullProfileAction,
	FullProfileWithRedeployAction,
	RebootAction,
	WalleHandleFailureAction,
}

func NewTaskType(taskType string, restrictions []string) *TaskType {
	return &TaskType{taskType, restrictions}
}

type TaskType struct {
	taskType     string
	restrictions []string
}

func (t *TaskType) Type() string {
	return t.taskType
}

func (t *TaskType) Restrictions() []string {
	return t.restrictions
}

func (t *TaskType) RestrictionsOk(fromSpec []string) bool {
	for _, r := range t.restrictions {
		if !slices.ContainsString(fromSpec, r) {
			return false
		}
	}
	return true
}

func ignoreCMSFromSafetyLevel(safetyLevel string) bool {
	switch safetyLevel {
	case SafetyLevelIgnoreAll:
		return true
	case SafetyLevelIgnoreCMS:
		return true
	case SafetyLevelNormal:
		return false
	}
	return false
}
