package tick

import (
	"context"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

/* Runs sync periodically */
type Periodic struct {
	name      string
	period    time.Duration
	s         Syncable
	failCount *unistat.Numeric
	l         *zap.Logger
}

func NewPeriodic(name string, period time.Duration, s Syncable) *Periodic {
	l, _ := zap.New(zap.ConsoleConfig(log.InfoLevel))
	f := unistat.NewNumeric(name+"-sync-fail", 10, aggr.Counter(), unistat.Sum)
	unistat.Register(f)
	return &Periodic{
		name:      name,
		period:    period,
		s:         s,
		failCount: f,
		l:         l,
	}
}

func (p *Periodic) Run(ctx context.Context) error {
	t := time.NewTicker(p.period)
	for {
		select {
		case <-ctx.Done():
			p.l.Warnf("Periodic sync '%s' stopped: %s", p.name, ctx.Err())
			return ctx.Err()
		case <-t.C:
			p.l.Infof("Start '%s'...", p.name)
			if err := p.s.Sync(ctx); err != nil {
				p.failCount.Update(1)
				p.l.Errorf("Sync '%s' failed: %s", p.name, err)
			}
		}
	}
}
