package validators

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/infra/maxwell/go/pkg/walle"
	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/slices"
)

func Valid(host *pb.Host, prj *pb.Project, spec *pb.Job_Spec, owner string, w walle.IClient) error {
	verdicts := make([]string, 0)
	if spec.Filters.Restrictions != nil {
		verdicts = append(verdicts, Restrictions(host, spec.Filters.Restrictions, prj)...)
	}
	if spec.Filters.Health != nil {
		verdicts = append(verdicts, Health(host, spec.Filters.Health)...)
	}
	if spec.Filters.Automation != nil {
		verdicts = append(verdicts, Automation(spec.Filters.Automation, prj)...)
	}
	verdicts = append(verdicts, Owner(host, owner, w)...)
	if len(verdicts) > 0 {
		return fmt.Errorf("host failed spec validation: %s", strings.Join(verdicts, ", "))
	}
	return nil
}

func Restrictions(host *pb.Host, restrictions *pb.Filter_Restrictions, project *pb.Project) []string {
	errs := make([]string, 0)
	if restrictions.Check == nil {
		return errs
	}
	if host.Restrictions == nil {
		return errs
	}
	if slices.ContainsAnyString(host.Restrictions, restrictions.Check) {
		return append(errs, "host failed restrictions validation")
	}
	if slices.ContainsAnyString(project.DefaultRestrictions, restrictions.Check) {
		return append(errs, "host failed project restrictions validation")
	}
	return errs
}

func Health(host *pb.Host, health *pb.Filter_Health) []string {
	var errs []string
	if host.Status != "ready" {
		return append(errs, fmt.Sprintf("host.status='%s' but should be 'ready'", host.Status))
	}
	errs = append(errs, filterByFailedHealth(host, health)...)
	errs = append(errs, filterByOkHealth(host, health)...)
	return errs
}

func filterByFailedHealth(host *pb.Host, health *pb.Filter_Health) []string {
	if health.FailedLogic == "OR" {
		for _, s := range health.Failed {
			if host.Health[s] == "failed" {
				return nil
			}
		}
		return []string{fmt.Sprintf("host does not satisfy (%s) == failed", strings.Join(health.Failed, " || "))}
	} else {
		for _, s := range health.Failed {
			if host.Health[s] != "failed" {
				return []string{fmt.Sprintf("host does not satisfy (%s) == failed", strings.Join(health.Failed, " && "))}
			}
		}
	}
	return nil
}

func filterByOkHealth(host *pb.Host, health *pb.Filter_Health) []string {
	for _, s := range health.Ok {
		if host.Health[s] == "failed" {
			return []string{fmt.Sprintf("host does not satisfy (%s) == ok", strings.Join(health.Ok, " && "))}
		}
	}
	return nil
}

func Automation(automation *pb.Filter_Automation, project *pb.Project) []string {
	errs := make([]string, 0)
	if !automation.Enabled {
		return errs
	}
	if project.DnsAutomation == pb.Project_DISABLED {
		errs = append(errs, "host failed validation: dns_automation disabled")
	}
	if project.HealingAutomation == pb.Project_DISABLED {
		errs = append(errs, "host failed validation: healing_automation disabled")
	}
	return errs
}

func Owner(host *pb.Host, owner string, w walle.IClient) []string {
	errs := make([]string, 0)
	if owner == "specman" || owner == "insecure" {
		return errs
	}
	isOwner, err := w.IsProjectOwner(host.Project, owner)
	if err != nil {
		return append(errs, err.Error())
	}
	if !isOwner {
		return append(errs, owner+" is not allowed to process this host")
	}
	return errs
}
