package validators

import (
	"fmt"
	"strings"
	"time"

	"github.com/golang/protobuf/ptypes"

	"a.yandex-team.ru/library/go/slices"

	"a.yandex-team.ru/infra/maxwell/go/internal/tasks"
	"a.yandex-team.ru/infra/maxwell/go/proto"
)

type Verdict struct {
	Reason string
}

var safetyLevels = []string{tasks.SafetyLevelNormal, tasks.SafetyLevelIgnoreCMS, tasks.SafetyLevelIgnoreAll}
var sshMethodAllowed = []string{tasks.SSHForbid, tasks.SSHOnly, tasks.SSHFallback}
var allowedGroups = []string{"location", "rack"}
var allowedOrder = []string{"location", "rack", "hostname", "movability"}

func SpecIsValid(spec *pb.Job_Spec) error {
	errs := make([]string, 0)
	errs = append(errs, groups(spec)...)
	errs = append(errs, order(spec)...)
	errs = append(errs, window(spec)...)
	errs = append(errs, overflowWindow(spec)...)
	errs = append(errs, safetyLevel(spec)...)
	errs = append(errs, name(spec)...)
	errs = append(errs, action(spec)...)
	errs = append(errs, restrictions(spec)...)
	errs = append(errs, methodSSH(spec)...)
	errs = append(errs, kexec(spec)...)
	errs = append(errs, applicabilitySSH(spec)...)
	errs = append(errs, mode(spec)...)
	errs = append(errs, timeouts(spec)...)
	errs = append(errs, filterHealthFailLogic(spec)...)
	errs = append(errs, unsafeOpsWindowSize(spec)...)
	if len(errs) > 0 {
		return fmt.Errorf("invalid spec: [%s]", strings.Join(errs, ", "))
	}
	return nil
}

func groups(spec *pb.Job_Spec) []string {
	if !slices.ContainsAllStrings(allowedGroups, spec.Group) {
		return []string{fmt.Sprintf("groups contains not allowed values, allowed: [%s]", strings.Join(allowedGroups, ", "))}
	}
	return nil
}

func order(spec *pb.Job_Spec) []string {
	if len(spec.Order) == 0 && spec.Source.GetWalle() != nil {
		return []string{"omitting order not allowed with wall-e source"}
	}
	if !slices.ContainsAllStrings(allowedOrder, spec.Order) {
		return []string{fmt.Sprintf("orders contains not allowed values, allowed: [%s]", strings.Join(allowedOrder, ", "))}
	}
	return nil
}

func safetyLevel(spec *pb.Job_Spec) []string {
	if !slices.ContainsString(safetyLevels, spec.SafetyLevel) {
		return []string{fmt.Sprintf("safety level should be one of: [%s]", strings.Join(safetyLevels, ","))}
	}
	return nil
}

const maxWindow = 300

func window(spec *pb.Job_Spec) []string {
	if spec.Window < 0 {
		return []string{"window should be greater then 0"}
	}
	if spec.SafetyLevel == tasks.SafetyLevelIgnoreAll {
		if spec.Window > 1 {
			return []string{"window should be less or eq then 1 for IGNORE_ALL"}
		}
		return nil
	}
	if spec.Window > maxWindow {
		return []string{fmt.Sprintf("window should be less then %d", maxWindow)}
	}
	return nil
}

func overflowWindow(spec *pb.Job_Spec) []string {
	if spec.SafetyLevel == tasks.SafetyLevelIgnoreAll {
		if spec.WindowOverflow == 0 {
			return []string{"window_overflow should be greater then 0 for IGNORE_ALL"}
		}
		return nil
	}
	if spec.WindowOverflow < 0 {
		return []string{"window_overflow should be greater then 0"}
	}
	if spec.WindowOverflow >= 10 {
		return []string{"window_overflow should be greater then 10"}
	}
	return nil
}

func name(spec *pb.Job_Spec) []string {
	if spec.Name == "" {
		return []string{"name should not be empty"}
	}
	return nil
}

func action(spec *pb.Job_Spec) []string {
	ok := false
	for _, action := range tasks.ValidActions {
		if action.Type() == spec.Action {
			ok = true
		}
	}
	if !ok {
		return []string{"invalid task_name"}
	}
	return nil
}

func restrictions(spec *pb.Job_Spec) []string {
	if spec.SafetyLevel == tasks.SafetyLevelIgnoreAll {
		return nil
	}
	if spec.Filters.Restrictions == nil {
		return []string{"restrictions filter should not be empty"}
	}
	action := tasks.UnknownAction
	for _, a := range tasks.ValidActions {
		if a.Type() == spec.Action {
			action = a
		}
	}
	if !action.RestrictionsOk(spec.Filters.Restrictions.Check) {
		return []string{fmt.Sprintf("restrictions is not valid, should be: %s", strings.Join(action.Restrictions(), ", "))}
	}
	return nil
}

// Verify SSH Method
func methodSSH(spec *pb.Job_Spec) []string {
	if spec.Ssh == "" {
		return nil
	}
	if !slices.ContainsString(sshMethodAllowed, spec.Ssh) {
		return []string{fmt.Sprintf("ssh method should be one of: [%s]", strings.Join(sshMethodAllowed, ","))}
	}
	return nil
}

func kexec(spec *pb.Job_Spec) []string {
	if !spec.Kexec {
		return nil
	}
	if spec.Action != tasks.RebootAction.Type() {
		return []string{fmt.Sprintf("kexec valid only for %s actions: but found %s", tasks.RebootAction.Type(), spec.Action)}
	}
	return nil
}

// Verify applicability SSH Method for task
func applicabilitySSH(spec *pb.Job_Spec) []string {
	if spec.Ssh != "" && spec.Action != tasks.RebootAction.Type() {
		return []string{fmt.Sprintf("ssh valid only for %s actions: but found %s", tasks.RebootAction.Type(), spec.Action)}
	}
	return nil
}

func mode(spec *pb.Job_Spec) []string {
	valid := []string{"follow", "script"}
	if !slices.ContainsString(valid, spec.Mode) {
		return []string{"invalid mode, should be one of: [%s]", strings.Join(valid, ", ")}
	}
	return nil
}

func timeouts(spec *pb.Job_Spec) []string {
	// Empty timeout it is ok. Means disabled.
	if spec.Timeouts == nil {
		return nil
	}
	t := spec.Timeouts
	errs := make([]string, 0)
	if t.Default != nil {
		if _, err := ptypes.Duration(t.Default); err != nil {
			errs = append(errs, err.Error())
		}
	}
	if t.Cms != nil {
		if _, err := ptypes.Duration(t.Cms); err != nil {
			errs = append(errs, err.Error())
		}
	}
	if t.Itdc != nil {
		if _, err := ptypes.Duration(t.Itdc); err != nil {
			errs = append(errs, err.Error())
		}
	}
	if t.Enforce != nil {
		if d, err := ptypes.Duration(t.Enforce); err != nil {
			errs = append(errs, err.Error())
		} else if d < time.Hour {
			errs = append(errs, fmt.Sprintf("enforce timeout too small: %s but min: %s", d, time.Hour))
		}
	}
	return errs
}

const (
	filterHealthLogicValueAND       = "AND"
	filterHealthLogicValueOR        = "OR"
	filterHealthLogicValueUndefined = ""
)

var allowedFilterHealthLogicValues = []string{filterHealthLogicValueAND, filterHealthLogicValueOR, filterHealthLogicValueUndefined}

func filterHealthFailLogic(s *pb.Job_Spec) []string {
	if s.Filters != nil && s.Filters.Health != nil {
		if !slices.ContainsString(allowedFilterHealthLogicValues, s.Filters.Health.FailedLogic) {
			return []string{
				fmt.Sprintf(
					"spec.filters.health.failed_logic value '%s' should be one of (%s) or empty",
					s.Filters.Health.FailedLogic, strings.Join(allowedFilterHealthLogicValues, ", "),
				),
			}
		}
	}
	return nil
}

const maxUnsafeOpsWindowSize = 5

func unsafeOpsWindowSize(s *pb.Job_Spec) []string {
	var rv []string
	if (s.SafetyLevel == tasks.SafetyLevelIgnoreCMS || s.SafetyLevel == tasks.SafetyLevelIgnoreAll) && s.Window > maxUnsafeOpsWindowSize {
		rv = append(rv, fmt.Sprintf("spec.window for unsafe ops must be not more than %d", maxUnsafeOpsWindowSize))
	}
	if s.Timeouts != nil && s.Timeouts.Enforce.IsValid() && s.Window > maxUnsafeOpsWindowSize {
		rv = append(rv, fmt.Sprintf("spec.window for tasks with enforce must be not more than %d", maxUnsafeOpsWindowSize))
	}
	return rv
}
