package workingset

import (
	"a.yandex-team.ru/infra/maxwell/go/internal/storages"
	"a.yandex-team.ru/infra/maxwell/go/internal/tasks"
	"a.yandex-team.ru/infra/maxwell/go/pkg/retry"
	"a.yandex-team.ru/infra/maxwell/go/pkg/walle"
	pb "a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
)

type Records []*pb.WorkingSetRecord

func (r Records) Created() Records {
	return r.Filter([]pb.Task_State{pb.Task_CREATED})
}

func (r Records) Running() Records {
	return r.Filter([]pb.Task_State{pb.Task_RUNNING})
}

func (r Records) SkipMuted() Records {
	return r.filter(func(t *pb.Task) bool { return !muted(t) })
}

func (r Records) Muted() Records {
	return r.filter(func(t *pb.Task) bool { return muted(t) })
}

func (r Records) Record(hostname string) *pb.WorkingSetRecord {
	rs := r.filter(func(t *pb.Task) bool { return t.Hostname == hostname })
	if len(rs) == 0 {
		return nil
	}
	return rs[0]
}

func (r Records) Filter(states []pb.Task_State) Records {
	return r.filter(func(t *pb.Task) bool {
		for _, state := range states {
			if t.State == state {
				return true
			}
		}
		return false
	})
}

func (r Records) FilterEnforce(states []pb.Task_Enforce) Records {
	return r.filter(func(t *pb.Task) bool {
		for _, state := range states {
			if t.Enforce == state {
				return true
			}
		}
		return false
	})
}

func (r Records) Tasks() []*pb.Task {
	t := make([]*pb.Task, len(r))
	for i, record := range r {
		t[i] = record.Task
	}
	return t
}

func muted(t *pb.Task) bool {
	return t.Mute != nil && t.Mute.Muted
}

func (r Records) filter(ok func(*pb.Task) bool) Records {
	filtered := make([]*pb.WorkingSetRecord, 0, len(r))
	for _, record := range r {
		if ok(record.Task) {
			filtered = append(filtered, record)
		}
	}
	return filtered
}

func New(name string, jobStorage storages.Job, hostsStorage storages.Hosts, spec *pb.Job_Spec) *WorkingSet {
	return &WorkingSet{name: name, jobStorage: jobStorage, hostsStorage: hostsStorage, spec: spec}
}

type WorkingSet struct {
	records      Records
	name         string
	jobStorage   storages.Job
	hostsStorage storages.Hosts
	spec         *pb.Job_Spec
}

func (ws *WorkingSet) Records() Records {
	return ws.records
}

// Filter removing records from working set if filter func (named ok) returns false
func (ws *WorkingSet) Filter(ok func(task *pb.Task) bool) ([]*pb.Task, error) {
	j := 0
	filtered := make([]*pb.Task, 0, len(ws.records))
	for i := 0; i < len(ws.records); i++ {
		record := ws.records[i]
		if ok(record.Task) {
			ws.records[j] = record
			j++
		} else {
			filtered = append(filtered, record.Task)
		}
	}
	for _, task := range filtered {
		if err := ws.hostsStorage.Pop(task.Hostname); err != nil {
			return nil, err
		}
	}
	// Cat filtered tail of records
	ws.records = ws.records[:j]
	return filtered, nil
}

func (ws *WorkingSet) Add(tasks ...*pb.WorkingSetRecord) {
	ws.records = append(ws.records, tasks...)
}

func (ws *WorkingSet) MuteTask(hostname, author string) {
	for i := 0; i < len(ws.records); i++ {
		r := ws.records[i]
		if r.Task.Hostname == hostname {
			r.Task.Mute = &pb.Task_Mute{
				Muted:  true,
				Author: author,
			}
			ws.records[i] = r
		}
	}
}

func (ws *WorkingSet) Update(w walle.IClient, l log.Logger) error {
	hostnames := make([]string, len(ws.records))
	for i, record := range ws.records {
		hostnames[i] = record.Task.Hostname
	}
	updated, err := w.GetHostsByNames(hostnames)
	if err != nil {
		return err
	}
	for i, record := range ws.records {
		host := updated[record.Task.Hostname]
		if err := w.UpdateLastTask(host); err != nil {
			l.Errorf("Failed to update last task time, resetting last task: %s", err)
		}
		tasks.UpdateState(record.Task, host, ws.spec)
		ws.records[i] = &pb.WorkingSetRecord{
			Task: record.Task,
			Host: host,
		}
	}
	return nil
}

func (ws *WorkingSet) Dump() error {
	ts := make([]*pb.Task, len(ws.records))
	hosts := make(map[string]*pb.Host)
	for i, record := range ws.records {
		hosts[record.Task.Hostname] = record.Host
		ts[i] = record.Task
	}
	if err := ws.hostsStorage.PutHosts(hosts); err != nil {
		return err
	}
	return ws.jobStorage.PutTasks(ts)
}

func (ws *WorkingSet) Restore() error {
	ts, err := ws.jobStorage.Tasks()
	if err != nil {
		return err
	}
	records := make([]*pb.WorkingSetRecord, len(ts))
	for i, task := range ts {
		host := &pb.Host{}
		err := retry.Retry(func() error {
			var err error
			host, err = ws.hostsStorage.Host(task.Hostname)
			return err
		})
		if err != nil {
			return err
		}
		records[i] = &pb.WorkingSetRecord{
			Task: task,
			Host: host,
		}
	}
	ws.records = records
	return nil
}
