package client

import (
	"encoding/json"
	"fmt"
	"google.golang.org/protobuf/encoding/protojson"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/core/log"
)

type Maxwell interface {
	PutJob(spec *pb.Job_Spec) (string, error)
	RestoreJob(id string) (string, error)
	StopJob(jobID string) (string, error)
	GetJobs() (*pb.Jobs, error)
	GetJob(string) (*pb.Job, error)
	DeleteJob(name string) error
	GetOAuth(code string) (string, error)
}

func CreateMaxwell(addr, accessToken string, l log.Logger) (Maxwell, error) {
	api := NewAPIClient(addr, accessToken)
	return &maxwell{
		api:         api,
		addr:        addr,
		accessToken: accessToken,
		l:           l,
	}, nil
}

type maxwell struct {
	api         *APIClient
	addr        string
	accessToken string
	l           log.Logger
}

func (m *maxwell) PutJob(spec *pb.Job_Spec) (string, error) {
	specB, err := proto.Marshal(spec)
	if err != nil {
		return "", err
	}
	resp, err := m.api.PostRequest("/api/jobs", specB, map[string]string{}, map[string]string{headerContentType: headerValueProtobuf})
	if err != nil {
		return "", err
	}
	return string(resp), nil
}

func (m *maxwell) RestoreJob(id string) (string, error) {
	resp, err := m.api.PostRequest(fmt.Sprintf("/api/jobs/%s/restore", id), make([]byte, 0), map[string]string{}, map[string]string{})
	if err != nil {
		return "", err
	}
	return string(resp), nil
}

func (m *maxwell) StopJob(jobID string) (string, error) {
	m.l.Infof("Stopping %s...", jobID)
	resp, err := m.api.DeleteRequest("/api/jobs/"+jobID, make([]byte, 0), map[string]string{}, map[string]string{})
	if err != nil {
		return "", err
	}
	return string(resp), nil
}

func (m *maxwell) GetJob(jobID string) (*pb.Job, error) {
	m.l.Infof("Getting %s...", jobID)
	resp, err := m.api.GetRequest("/api/jobs/"+jobID, map[string]string{}, map[string]string{})
	if err != nil {
		return nil, err
	}
	job := &pb.GetJobResponse{}
	err = protojson.Unmarshal(resp, job)
	if err != nil {
		return nil, err
	}
	return job.Job.Job, nil
}

func (m *maxwell) DeleteJob(name string) error {
	m.l.Infof("Deleting %s...", name)
	_, err := m.api.DeleteRequest(fmt.Sprintf("/api/jobs/%s/delete", name), make([]byte, 0), map[string]string{}, map[string]string{})
	return err
}

func (m *maxwell) GetJobs() (*pb.Jobs, error) {
	m.l.Infof("Listing jobs...")
	resp, err := m.api.GetRequest("/api/jobs", map[string]string{}, map[string]string{})
	if err != nil {
		return nil, err
	}
	respProto := &pb.ListJobsResponse{}
	if err := protojson.Unmarshal(resp, respProto); err != nil {
		return nil, err
	}
	m.l.Infof("Received %d jobs", len(respProto.Jobs))
	jobs := make([]*pb.Job, len(respProto.Jobs))
	for i, name := range respProto.Jobs {
		jobs[i], err = m.GetJob(name)
		if err != nil {
			return nil, err
		}
	}
	return &pb.Jobs{
		Jobs: jobs,
	}, nil
}

type OAuthResponse struct {
	TokenType    string `json:"token_type"`
	AccessToken  string `json:"access_token"`
	RefreshToken string `json:"refresh_token"`
}

func (m *maxwell) GetOAuth(code string) (string, error) {
	respBytes, err := m.api.PostRequest("/oauthCode", make([]byte, 0), map[string]string{}, map[string]string{"OAuthCode": code})
	if err != nil {
		return "", err
	}
	resp := &OAuthResponse{}
	err = json.Unmarshal(respBytes, resp)
	if err != nil {
		return "", err
	}
	return resp.AccessToken, nil
}
