package server

import (
	"a.yandex-team.ru/infra/maxwell/go/internal/leader"
	"context"
	"fmt"
	"go.etcd.io/etcd/clientv3"
	"os"
	"os/signal"
	"sync"
	"syscall"

	"a.yandex-team.ru/infra/maxwell/go/internal/job"
	"a.yandex-team.ru/infra/maxwell/go/pkg/nanny"
	"a.yandex-team.ru/infra/maxwell/go/pkg/walle"
	"a.yandex-team.ru/infra/maxwell/go/pkg/yp"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/unistat"
)

const (
	Addr = "[::]:80"
)

type MaxwellConfig struct {
	WalleToken   string
	NannyToken   string
	YpToken      string
	SpecsPath    string
	ClientID     string
	ClientSecret string
	InMemory     bool
	Addr         string
	Insecure     bool
	DryRun       bool
	Hostname     string
	Debug        bool
}

type Maxwell struct {
	Jobman        *job.Manager
	MaxwellServer *Server
	r             unistat.Registry
	l             log.Logger

	etcdClient *clientv3.Client
	hostname   string
}

func CreateMaxwell(config *MaxwellConfig) (*Maxwell, error) {
	cfg := zap.ConsoleConfig(log.InfoLevel)
	l, _ := zap.New(cfg)
	api := walle.NewAPIClient(walle.URLProduction, config.WalleToken)
	w := walle.NewClient(api)
	n, err := nanny.NewClient(nanny.APIURLProd, config.NannyToken)
	if err != nil {
		return nil, err
	}
	y, err := yp.GetYpClientsPool(l, yp.YpClusters, config.YpToken)
	if err != nil {
		return nil, err
	}
	etcdClient, err := clientv3.New(
		clientv3.Config{
			Endpoints:           []string{"127.0.0.1:2379"},
			MaxCallSendMsgSize:  100 * 1024 * 1024, // 100MB
			LogConfig:           &cfg,
			PermitWithoutStream: false,
		},
	)
	if err != nil {
		return nil, err
	}
	jm, err := job.NewManager(etcdClient, config.InMemory, w, l, config.DryRun, n, y)
	if err != nil {
		l.Errorf("%s", err)
		return nil, err
	}
	serv := CreateServer(l, jm, config.Addr, config.ClientID, config.ClientSecret, config.Insecure, config.Debug)
	return &Maxwell{
		Jobman:        jm,
		l:             l,
		MaxwellServer: serv,
		etcdClient:    etcdClient,
		hostname:      config.Hostname,
	}, nil
}

func (m *Maxwell) Start() error {
	ctx, cancel := context.WithCancel(context.Background())
	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGINT, syscall.SIGTERM)
	go func() {
		s := <-sigs
		m.l.Infof("Received %s, closing", s)
		cancel()
	}()
	errCh := make(chan error)
	wg := sync.WaitGroup{}
	wg.Add(3)
	go func() {
		err := m.MaxwellServer.Start(ctx)
		if err != nil {
			errCh <- fmt.Errorf("maxwell_server: %w", err)
		}
		wg.Done()
	}()
	go func() {
		if err := m.Jobman.RunMonitoring(ctx); err != nil {
			errCh <- fmt.Errorf("jobman: %w", err)
		}
		wg.Done()
	}()
	go func() {
		leader.LockAndRun(ctx, m.l, m.etcdClient, "/manager/master", m.hostname, func(ctx context.Context) {
			if err := m.Jobman.RunMaster(ctx); err != nil {
				errCh <- fmt.Errorf("jobman: %w", err)
			}
		})
		wg.Done()
	}()
	done := make(chan bool)
	go func() {
		wg.Wait()
		done <- true
	}()
	select {
	case err := <-errCh:
		m.l.Fatal(err.Error())
		return err
	case <-done:
		return nil
	}
}
