package server

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/infra/maxwell/go/internal/api"
	"a.yandex-team.ru/infra/maxwell/go/internal/httpserver"
	"a.yandex-team.ru/infra/maxwell/go/internal/job"
	"a.yandex-team.ru/infra/maxwell/ui"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"

	"github.com/go-chi/chi/v5/middleware"
)

type Server struct {
	jobm *job.Manager
	serv *httpserver.Server
	grpc *api.JobmanAPI
	l    log.Logger
}

func initBB() (blackbox.Client, error) {
	z, _ := zap.New(zap.ConsoleConfig(log.InfoLevel))
	return httpbb.NewIntranet(
		httpbb.WithLogger(z),
	)
}

func CreateServer(l log.Logger, jm *job.Manager, addr, clientID, clientSecret string, insecure bool, debug bool) *Server {
	bb, err := initBB()
	if err != nil {
		panic(err)
	}
	hs := httpserver.NewHTTP(addr)
	unistatAPI := api.NewUnistat(l)
	jmAPI := api.NewJobmanAPI(jm, bb, insecure)
	uiAPI := ui.NewHandler()
	auth := api.NewLoginHandler(bb, clientID, clientSecret, insecure)
	unistatAPI.Register(hs.Mux())
	jmAPI.Register(hs.Mux())
	auth.Register(hs.Mux())
	uiAPI.Register(hs.Mux())
	if debug {
		hs.Mux().Mount("/debug", middleware.Profiler())
	}
	return &Server{
		jobm: jm,
		serv: hs,
		grpc: jmAPI,
		l:    l,
	}
}

func (s *Server) Start(ctx context.Context) error {
	ctx, cancel := context.WithCancel(ctx)
	errCh := make(chan error)
	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGINT, syscall.SIGTERM)
	go func() {
		<-sigs
		cancel()
	}()
	go func() {
		if err := s.serv.Run(ctx, s.l); err != http.ErrServerClosed {
			errCh <- fmt.Errorf("http server: %w", err)
		}
		fmt.Println("http server closed")
	}()
	return nil
}
