package nanny

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

const (
	retryCount      = 3
	readTimeout     = 120 * time.Second
	headerValueJSON = "application/json"
	userAgent       = "Maxwell.Nanny.Client"

	APIURLProd     = "https://nanny.yandex-team.ru"
	aPIPrefix      = "/api/repo"
	paginatorLimit = 200

	getReplicationPolicyEndpoint    = "/GetReplicationPolicy/"
	listSummariesEndpoint           = "/ListSummaries/"
	listReplicationPoliciesEndpoint = "/ListReplicationPolicies/"
)

const (
	ServiceStatusOnline       = "ONLINE"
	ServiceStatusOffline      = "OFFLINE"
	PolicyMessageAllPodsAreOk = "all pods are ok"
	PolicyStatusIdle          = "idle"
)

var ErrHTTPCode = errors.New("wrong http response status code")

type client struct {
	c *resty.Client
}

// NewClient function returns new nanny rpc http client
func NewClient(apiServer string, accessToken string) (*client, error) {

	client := client{
		c: resty.New(),
	}
	if accessToken != "" {
		client.
			c.SetHeaders(map[string]string{
			"Content-Type":  headerValueJSON,
			"User-Agent":    userAgent,
			"Authorization": fmt.Sprintf("OAuth %s", accessToken),
		})
	} else {
		return nil, errors.New("token not set or empty")
	}
	client.c.SetRetryCount(retryCount).SetTimeout(readTimeout).SetHostURL(apiServer)
	return &client, nil
}

type Client interface {
	ListSummaries() ([]*ServiceSummary, error)
	ListReplicationPolicies() (map[string]*ServiceReplicationStatus, error)
	GetReplicationPolicy(serviceID string) (*ServiceReplicationStatus, error)
}

// List Summaries returns nanny service summary info
func (c *client) ListSummaries() ([]*ServiceSummary, error) {
	var resp ListSummariesResponse
	r, err := c.c.R().
		Get(aPIPrefix + listSummariesEndpoint)
	if err != nil {
		return nil, err
	}
	// non statusok code is an error for this api call
	if r.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("%w (http code: '%d')", ErrHTTPCode, r.StatusCode())
	}
	err = json.Unmarshal(r.Body(), &resp)
	if err != nil {
		return nil, err
	}
	return resp.Value, nil
}

type ServiceSummary struct {
	Summary    string `json:"summary,omitempty"`
	ServicedID string `json:"serviceid,omitempty"`
}

type ListSummariesResponse struct {
	Total int               `json:"total,omitempty"`
	Value []*ServiceSummary `json:"value,omitempty"`
}

// GetReplicationPolicy returns replication policy for nanny service
func (c *client) GetReplicationPolicy(serviceID string) (*ServiceReplicationStatus, error) {
	var resp ServiceReplicationResponse
	data := []byte(fmt.Sprintf("policy_id: %s", serviceID))
	r, err := c.c.R().
		SetBody(data).
		Post(aPIPrefix + getReplicationPolicyEndpoint)
	if err != nil {
		return nil, err
	}
	// non statusok code is an error for this api call
	if r.StatusCode() != 200 {
		return nil, fmt.Errorf("%w (http code: '%d')", ErrHTTPCode, r.StatusCode())
	}
	err = json.Unmarshal(r.Body(), &resp)
	if err != nil {
		return nil, err
	}
	return resp.Policy.Status, nil
}

// ListReplicationPolicies returns replication policies for all nanny services
func (c *client) ListReplicationPolicies() (map[string]*ServiceReplicationStatus, error) {
	p := map[string]*ServiceReplicationStatus{}
	l := paginatorLimit
	s := 0
	// paginated response need iterate over all response pages
	for ok := true; ok; ok = l == paginatorLimit {
		resp := ListReplicationResponse{}
		data := map[string]string{
			"limit": fmt.Sprintf("%d", l),
			"skip":  fmt.Sprintf("%d", s*paginatorLimit),
		}
		r, err := c.c.R().SetQueryParams(data).Get(aPIPrefix + listReplicationPoliciesEndpoint)
		if err != nil {
			return nil, err
		}
		// non statusok code is an error for this api call
		if r.StatusCode() != 200 {
			return nil, fmt.Errorf("%w (http code: '%d')", ErrHTTPCode, r.StatusCode())
		}
		if err := json.Unmarshal(r.Body(), &resp); err != nil {
			return nil, err
		}
		for _, s := range resp.Policy {
			if s.Status != nil {
				p[s.Meta.ID] = s.Status
			}
		}
		l = len(resp.Policy)
		s = s + 1
	}
	return p, nil
}

type ServiceReplicationResponse struct {
	Policy *ServiceReplicationPolicy `json:"policy,omitempty"`
}

type ServiceReplicationPolicy struct {
	Status *ServiceReplicationStatus `json:"status,omitempty"`
}

type ServiceReplicationStatus struct {
	Status  string `json:"status,omitempty"`
	Message string `json:"message,omitempty"`
}

type ListReplicationResponse struct {
	Policy []*ListReplicationPolicy `json:"policy,omitempty"`
}

type ListReplicationPolicy struct {
	Status *ServiceReplicationStatus `json:"status,omitempty"`
	Meta   *ListReplicationMeta      `json:"meta,omitempty"`
}

type ListReplicationMeta struct {
	ID string `json:"id,omitempty"`
}

func CompileServices(policies map[string]*ServiceReplicationStatus, summaries []*ServiceSummary) map[string]*pb.NannyService {
	result := make(map[string]*pb.NannyService)
	for k, v := range policies {
		service := &pb.NannyService{}
		service.MetaId = k
		service.Summary = ""
		service.ReplicationPolicyStatus = v.Status
		service.ReplicationPolicyMessage = v.Message
		for _, s := range summaries {
			if s.ServicedID == k {
				service.Summary = s.Summary
				break
			}
		}
		movable := true
		if service.Summary != ServiceStatusOffline && service.Summary != ServiceStatusOnline {
			movable = false
		}
		if strings.ToLower(service.ReplicationPolicyMessage) != PolicyMessageAllPodsAreOk {
			movable = false
		}
		if strings.ToLower(service.ReplicationPolicyStatus) != PolicyStatusIdle {
			movable = false
		}
		service.Moveable = movable
		result[k] = service
	}
	return result
}
