package nanny

import (
	"bytes"
	"github.com/stretchr/testify/require"
	"io"
	"net/http"
	"testing"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

type transportMock struct {
	host     string
	url      string
	response []byte
}

type readCloser struct {
	r io.Reader
}

func (r *readCloser) Read(p []byte) (n int, err error) {
	return r.r.Read(p)
}

func (r *readCloser) Close() error {
	return nil
}

func (t *transportMock) RoundTrip(request *http.Request) (*http.Response, error) {
	rsp := &http.Response{}
	if request.URL.Host != t.host || request.URL.Path != t.url {
		rsp.StatusCode = http.StatusNotFound
	} else {
		rsp.StatusCode = http.StatusOK
		rsp.Body = &readCloser{r: bytes.NewReader(t.response)}
	}
	return rsp, nil
}

// TestNewClient function test that creation client without token raises error and returns nil client
func TestNewClient(t *testing.T) {
	exp, err := NewClient("http://example", "")
	require.Error(t, err)
	require.Equal(t, exp, (*client)(nil))
}

func TestClient_GetReplicationPolicy(t *testing.T) {
	c, _ := NewClient("http://someurl", "sometoken")
	s := "xxx"
	b := []byte(`{ "policy": {"status": {"status": "xxx", "message": "x"}}}`)
	c.c.SetTransport(&transportMock{host: "someurl", url: aPIPrefix + getReplicationPolicyEndpoint, response: b})
	exp, err := c.GetReplicationPolicy(s)
	require.Equal(t, err, nil)
	require.Equal(t, exp, &ServiceReplicationStatus{Status: "xxx", Message: "x"})
}

func TestClient_ListSummaries(t *testing.T) {
	c, _ := NewClient("http://someurl", "sometoken")
	b := []byte(`{ "total": 1, "value": [{"summary": "xxx", "serviceid": "x"}] }`)
	c.c.SetTransport(&transportMock{host: "someurl", url: aPIPrefix + listSummariesEndpoint, response: b})
	exp, err := c.ListSummaries()
	require.Equal(t, err, nil)
	require.Equal(t, exp, []*ServiceSummary{&ServiceSummary{Summary: "xxx", ServicedID: "x"}})
}

func TestCompileServices(t *testing.T) {
	require.Equal(t, CompileServices(map[string]*ServiceReplicationStatus{"nanny1": &ServiceReplicationStatus{Status: PolicyStatusIdle, Message: PolicyMessageAllPodsAreOk}}, []*ServiceSummary{{ServicedID: "nanny1", Summary: ServiceStatusOnline}}),
		map[string]*pb.NannyService{"nanny1": &pb.NannyService{MetaId: "nanny1", Summary: ServiceStatusOnline, ReplicationPolicyStatus: PolicyStatusIdle, ReplicationPolicyMessage: PolicyMessageAllPodsAreOk, Moveable: true}})

	// check case for deploing nanny service
	require.Equal(t, CompileServices(map[string]*ServiceReplicationStatus{"nanny1": &ServiceReplicationStatus{Status: PolicyStatusIdle, Message: PolicyMessageAllPodsAreOk}}, []*ServiceSummary{{ServicedID: "nanny1", Summary: "DEPLOYING"}}),
		map[string]*pb.NannyService{"nanny1": &pb.NannyService{MetaId: "nanny1", Summary: "DEPLOYING", ReplicationPolicyStatus: PolicyStatusIdle, ReplicationPolicyMessage: PolicyMessageAllPodsAreOk, Moveable: false}})

}
