package walle

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"
)

type APIClient struct {
	url        string
	headers    http.Header
	httpClient http.Client
}

func NewAPIClient(url string, accessToken string) *APIClient {
	if url[len(url)-1] == '/' {
		url = url[:len(url)-1]
	}
	url += fmt.Sprintf("/%s", apiVersion)
	client := APIClient{
		headers: http.Header{
			"Content-Type": []string{headerValueJSON},
			"User-Agent":   []string{userAgent},
		},
		url: url,
		httpClient: http.Client{
			Timeout: 5 * time.Second,
		},
	}
	if accessToken != "" {
		client.headers.Add("Authorization", fmt.Sprintf("OAuth %s", accessToken))
	}
	return &client
}

func (c *APIClient) request(httpMethod string, uri string, data []byte, params map[string]string) (resp []byte, err error) {
	httpReq, err := http.NewRequest(httpMethod, c.url+uri, bytes.NewBuffer(data))
	if err != nil {
		return nil, err
	}
	httpReq.Header = c.headers
	values := httpReq.URL.Query()
	for k, v := range params {
		if v == "" {
			continue
		}
		values.Add(k, v)
	}
	values.Add("strict", "true")
	httpReq.URL.RawQuery = values.Encode()
	httpResp, err := c.httpClient.Do(httpReq)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = httpResp.Body.Close()
	}()
	resp, err = ioutil.ReadAll(httpResp.Body)
	if err != nil {
		return nil, fmt.Errorf("failed to read response: %s", err.Error())
	}
	if httpResp.StatusCode != 200 {
		return nil, fmt.Errorf("bad response code: %d, message: %s", httpResp.StatusCode, string(resp))
	}
	return resp, nil
}

func (c *APIClient) GetRequest(uri string, params map[string]string) (resp []byte, err error) {
	return c.request(http.MethodGet, uri, make([]byte, 0), params)
}

func (c *APIClient) PutRequest(uri string, data []byte, params map[string]string) (resp []byte, err error) {
	return c.request(http.MethodPut, uri, data, params)
}

func (c *APIClient) PostRequest(uri string, data []byte, params map[string]string) (resp []byte, err error) {
	return c.request(http.MethodPost, uri, data, params)
}

func (c *APIClient) DeleteRequest(uri string, data []byte, params map[string]string) (resp []byte, err error) {
	return c.request(http.MethodDelete, uri, data, params)
}

func (c *APIClient) PatchRequest(uri string, data []byte, params map[string]string) (resp []byte, err error) {
	return c.request(http.MethodPatch, uri, data, params)
}
