package walle

import (
	"encoding/json"
	"fmt"
	"strconv"
	"strings"
	"time"

	"github.com/golang/protobuf/ptypes"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
	"a.yandex-team.ru/library/go/slices"
)

type Client struct {
	APIClient *APIClient
}

type IClient interface {
	GetHost(hostname string) (*pb.Host, error)
	GetProject(name string) (*pb.Project, error)
	GetHosts(req *GetHostsRequest) (*GetHostsResponse, error)
	GetHostsByNames([]string) (map[string]*pb.Host, error)
	ProfileHost(req *ProfileHostRequest) ([]byte, error)
	RedeployHost(req *RedeployHostRequest) ([]byte, error)
	RebootHost(req *RebootHostRequest) ([]byte, error)
	RebootKexec(hostname string, req *RebootKexecHostBody) error
	GetHealthCheck(fqdn string, checkName string) (*HealthCheck, error)
	FirmwareProblems(fqdn string) ([]string, error)
	GetHealthChecks(hostname string, checks []string) (*HealthChecksResp, error)
	IsProjectOwner(project string, owner string) (bool, error)
	UpdateLastTask(host *pb.Host) error
	CancelTask(hostname string) error
	HandleFailure(hostname string, req *HandleFailureRequest) error
}

func NewClient(api *APIClient) *Client {
	return &Client{
		APIClient: api,
	}
}

func (c *Client) GetHost(hostname string) (*pb.Host, error) {
	uri := fmt.Sprintf("/hosts/%s", hostname)
	fields := []string{
		"restrictions",
		"health.check_statuses",
		"name",
		"status",
		"task.status",
		"task.status_message",
		"project",
		"location",
		"location.country",
		"location.city",
		"location.queue",
		"location.rack",
		"location.short_datacenter_name",
		"tags",
		"ticket",
	}
	params := map[string]string{"fields": strings.Join(fields, ",")}
	params["resolve_tags"] = "true"
	resp := &HostStatus{}
	respBytes, err := c.APIClient.GetRequest(uri, params)
	if err != nil {
		return nil, err
	}
	err = json.Unmarshal(respBytes, &resp)
	if err != nil {
		return nil, err
	}
	host := hostFromStatus(resp)
	return host, nil
}

func (c *Client) GetHosts(req *GetHostsRequest) (*GetHostsResponse, error) {
	fields := []string{
		"restrictions",
		"health.check_statuses",
		"name",
		"status",
		"task.status",
		"task.status_message",
		"project",
		"location",
		"location.country",
		"location.city",
		"location.queue",
		"location.rack",
		"location.short_datacenter_name",
		"tags",
		"ticket",
	}
	params := &GetHostsRequestParams{
		Fields:           fields,
		Cursor:           req.Cursor,
		Limit:            int64(req.FetchLimit),
		Status:           "ready",
		Project:          req.Project,
		Tags:             strings.Join(req.Tags, ","),
		PhysicalLocation: req.Location,
		ResolveTags:      "true",
	}
	resp := HostsStatus{}
	respBytes, err := c.APIClient.GetRequest("/hosts", params.Format())
	if err != nil {
		return nil, err
	}
	err = json.Unmarshal(respBytes, &resp)
	hosts := make([]*pb.Host, len(resp.Result))
	for i := 0; i < len(hosts); i++ {
		hosts[i] = hostFromStatus(resp.Result[i])
	}
	return &GetHostsResponse{
		Hosts:      hosts,
		Total:      resp.Total,
		NextCursor: resp.NextCursor,
	}, err
}

func (c *Client) GetHostsByNames(names []string) (map[string]*pb.Host, error) {
	if len(names) > 1000 {
		return nil, fmt.Errorf("too many hosts, max 1000")
	}
	fields := []string{
		"restrictions",
		"health.check_statuses",
		"name",
		"status",
		"task.status",
		"task.status_message",
		"project",
		"location",
		"location.country",
		"location.city",
		"location.queue",
		"location.rack",
		"location.short_datacenter_name",
		"tags",
		"ticket",
	}
	params := (&GetHostsRequestParams{
		Fields:      fields,
		Limit:       1000,
		ResolveTags: "true",
	}).Format()
	resp := &HostsStatus{}
	data, err := json.Marshal(map[string][]string{
		"names": names,
	})
	if err != nil {
		return nil, err
	}
	respBytes, err := c.APIClient.PostRequest("/get-hosts", data, params)
	if err != nil {
		return nil, err
	}
	err = json.Unmarshal(respBytes, resp)
	hosts := make(map[string]*pb.Host)
	for i := 0; i < len(resp.Result); i++ {
		host := hostFromStatus(resp.Result[i])
		hosts[resp.Result[i].Name] = host
	}
	// Fill all lost names with empty &pb.Host{}.
	// Empty value better then nil.
	// We'll panic on nil value, on empty fields we'll show problem in UI
	// TODO: autoremoving invalid hosts (must be safe operation, show removing status in UI, or other notification)
	// For now do not remove, just show empty host in UI in invalid status
	for _, name := range names {
		if _, ok := hosts[name]; !ok {
			hosts[name] = &pb.Host{
				Health:           make(map[string]string),
				Location:         &pb.Host_Location{},
				Restrictions:     make([]string, 0),
				FirmwareProblems: make([]string, 0),
				Status:           "invalid",
			}
		}
	}
	return hosts, err
}

func (c *Client) GetHealthChecks(hostname string, checks []string) (*HealthChecksResp, error) {
	req := &GetHealthChecksRequest{
		Names: []string{hostname},
		Type:  checks,
	}
	resp := HealthChecksResp{}
	params := map[string]string{
		"fqdn":   strings.Join(req.Names, ","),
		"limit":  fmt.Sprintf(req.Limit),
		"offset": fmt.Sprintf(req.Offset),
		"type":   strings.Join(req.Type, ","),
	}
	respBytes, err := c.APIClient.GetRequest("/health-checks", params)
	if err != nil {
		return nil, err
	}
	err = json.Unmarshal(respBytes, &resp)
	return &resp, err
}

func (c *Client) FirmwareProblems(hostname string) ([]string, error) {
	walleFirmware, err := c.GetHealthCheck(hostname, "walle_firmware")
	if err != nil {
		return nil, err
	}
	mj, _ := json.Marshal(walleFirmware.Metadata)
	var m WalleFirmwareMetadata
	if err := json.Unmarshal(mj, &m); err != nil {
		return nil, err
	}
	return m.Type, nil
}

func (c *Client) GetHealthCheck(fqdn string, checkName string) (*HealthCheck, error) {
	healthChecks, err := c.GetHealthChecks(fqdn, []string{checkName})
	if err != nil {
		return nil, err
	}
	for _, c := range healthChecks.Result {
		if c.Type == checkName {
			return &c, nil
		}
	}
	return nil, fmt.Errorf("check name not found")
}

func (c *Client) ProfileHost(req *ProfileHostRequest) ([]byte, error) {
	params := req.Params.Format()
	data, err := json.Marshal(req.Body)
	uri := fmt.Sprintf("/hosts/%s/profile", req.HostID)
	if err != nil {
		return make([]byte, 0), err
	}
	return c.APIClient.PostRequest(uri, data, params)
}

func (c *Client) RedeployHost(req *RedeployHostRequest) ([]byte, error) {
	params := req.Params.Format()
	data, err := json.Marshal(req.Body)
	uri := fmt.Sprintf("/hosts/%s/redeploy", req.HostID)
	if err != nil {
		return nil, err
	}
	return c.APIClient.PostRequest(uri, data, params)
}

func (c *Client) RebootHost(req *RebootHostRequest) ([]byte, error) {
	params := req.Params.Format()
	data, err := json.Marshal(req.Body)
	uri := fmt.Sprintf("/hosts/%s/reboot", req.HostID)
	if err != nil {
		return make([]byte, 0), err
	}
	return c.APIClient.PostRequest(uri, data, params)
}

func (c *Client) RebootKexec(hostname string, req *RebootKexecHostBody) error {
	data, err := json.Marshal(req)
	uri := fmt.Sprintf("/hosts/%s/kexec_reboot", hostname)
	if err != nil {
		return err
	}
	_, err = c.APIClient.PostRequest(uri, data, make(map[string]string))
	return err
}

func (c *Client) ValidateRestrictions(hostname string, checkRestrictions []string) (bool, error) {
	restrictions, err := c.getRestrictions(hostname)
	if err != nil {
		return false, err
	}
	if checkRestrictions == nil {
		return true, nil
	}
	if restrictions == nil {
		return true, nil
	}
	if slices.ContainsAnyString(restrictions, checkRestrictions) {
		return false, nil
	}
	return true, nil
}

func (c *Client) IsProjectOwner(project, owner string) (bool, error) {
	uri := fmt.Sprintf("/projects/%s/is_project_owner/%s", project, owner)
	resp, err := c.APIClient.GetRequest(uri, make(map[string]string))
	if err != nil {
		return false, err
	}
	respJSON := make(map[string]bool)
	err = json.Unmarshal(resp, &respJSON)
	if err != nil {
		return false, err
	}
	return respJSON["is_owner"], nil
}

func (c *Client) GetProject(name string) (*pb.Project, error) {
	uri := fmt.Sprintf("/projects/%s?fields=%s",
		name,
		strings.Join([]string{"default_host_restrictions", "dns_automation", "healing_automation"}, ","),
	)
	resp, err := c.APIClient.GetRequest(uri, make(map[string]string))
	if err != nil {
		return nil, err
	}
	respPB := &ProjectResp{}
	err = json.Unmarshal(resp, &respPB)
	if err != nil {
		return nil, err
	}
	prj := &pb.Project{
		DefaultRestrictions: respPB.DefaultHostRestrictions,
	}
	if respPB.HealingAutomation.Enabled {
		prj.HealingAutomation = pb.Project_ENABLED
	} else {
		prj.HealingAutomation = pb.Project_DISABLED
	}
	if respPB.DNSAutomation.Enabled {
		prj.DnsAutomation = pb.Project_ENABLED
	} else {
		prj.DnsAutomation = pb.Project_DISABLED
	}
	return prj, nil
}

func (c *Client) UpdateLastTask(host *pb.Host) error {
	er := func(err error) error {
		// We need this field consistent
		// Reset on failure, marking as failure
		host.LastTask = nil
		return err
	}
	uri := fmt.Sprintf("/audit-log?fields=time,reason&host_name=%s&limit=1", host.Hostname)
	resp, err := c.APIClient.GetRequest(uri, make(map[string]string))
	if err != nil {
		return er(err)
	}
	respPB := &AuditLogResp{}
	if err := json.Unmarshal(resp, &respPB); err != nil {
		return er(err)
	}
	if len(respPB.Result) == 0 {
		return er(fmt.Errorf("audit log empty"))
	}
	r := respPB.Result[0]
	last := time.Unix(int64(r.Time), 0)
	plast, err := ptypes.TimestampProto(last)
	if err != nil {
		return er(err)
	}
	if host.LastTask == nil {
		host.LastTask = &pb.Host_Task{}
	}
	host.LastTask.Time = plast
	host.LastTask.Description = r.Reason
	return nil
}

func (c *Client) CancelTask(hostname string) error {
	uri := fmt.Sprintf("/hosts/%s/cancel-task", hostname)
	body, _ := json.Marshal(map[string]string{"reason": "Maxwell: enforce task"})
	_, err := c.APIClient.PostRequest(uri, body, make(map[string]string))
	return err
}

func (c *Client) getRestrictions(hostname string) ([]string, error) {
	h, err := c.GetHost(hostname)
	if err != nil {
		return nil, err
	}
	return h.Restrictions, nil
}

func (c *Client) HandleFailure(hostname string, req *HandleFailureRequest) error {
	url := fmt.Sprintf("/hosts/%s/handle-failure", hostname)
	buf, err := json.Marshal(req)
	if err != nil {
		return fmt.Errorf("failed to marshal HandleFailureRequest: %w", err)
	}
	if _, err := c.APIClient.PostRequest(url, buf, nil); err != nil {
		return fmt.Errorf("failed to issue handle-failure request for %s with body '%s': %w", hostname, buf, err)
	}
	return nil
}

// InferYPClusterByTag return ypcluster for yp project and yp location cluster for gencfg project
func InferYPClusterByTag(tags []string, dc string) string {
	var cluster string
	for _, t := range tags {
		if strings.HasPrefix(t, ypClusterTag) {
			if cluster != "" {
				cluster = ""
				break
			}
			res, here := ypMasterTagMap[t]
			if !here {
				break
			}
			cluster = res
		}
	}
	if cluster == "" {
		for _, p := range ypMasterDC {
			if dc == p {
				cluster = dc + ".yp.yandex.net"
				break
			}
		}
	}
	return cluster

}

func hostFromStatus(status *HostStatus) *pb.Host {
	task := ""
	if status.Task != nil {
		task = status.Task.Status
	}
	return &pb.Host{
		Health: status.Health.CheckStatuses,
		Location: &pb.Host_Location{
			Country:             status.Location.Country,
			City:                status.Location.City,
			Rack:                status.Location.Rack,
			Queue:               status.Location.Queue,
			ShortDatacenterName: status.Location.ShortDatacenterName,
		},
		Restrictions:     status.Restrictions,
		FirmwareProblems: nil, // TODO:
		Project:          status.Project,
		Hostname:         status.Name,
		Status:           status.Status,
		Task:             task,
		WalleTags:        status.Tags,
		Ticket:           status.Ticket,
		YpCluster:        InferYPClusterByTag(status.Tags, status.Location.ShortDatacenterName),
	}
}

type AuditLogResp struct {
	Result []AuditLog `json:"result,omitempty"`
}

type AuditLog struct {
	Time   float64 `json:"time,omitempty"`
	Reason string  `json:"reason,omitempty"`
}

type GetHostRequest struct {
	HostID                     string
	ResolveDeployConfiguration bool
	Fields                     []string
	Data                       []byte
}

type GetHostsRequest struct {
	Cursor     int
	FetchLimit int
	Project    string
	Tags       []string
	Location   string
}

type GetHealthChecksRequest struct {
	Names  []string
	Limit  string
	Offset string
	Type   []string
}

type GetProjectRequest struct {
	Fields    []string
	ProjectID string
}

type ProfileHostRequest struct {
	HostID string
	Params RedeployHostParams
	Body   ProfileHostBody
}

type RedeployHostBody struct {
	Check                bool   `json:"check,omitempty"`
	Config               string `json:"config,omitempty"`
	DeployConfigPolicy   string `json:"deploy_config_policy,omitempty"`
	DisableAdminRequests bool   `json:"disable_admin_requests,omitempty"`
	IgnoreCms            bool   `json:"ignore_cms,omitempty"`
	Network              string `json:"network,omitempty"`
	Provisioner          string `json:"provisioner,omitempty"`
	Reason               string `json:"reason,omitempty"`
	Tags                 string `json:"tags,omitempty"`
	WithAutoHealing      bool   `json:"with_auto_healing,omitempty"`
}

type RebootHostBody struct {
	Check                bool   `json:"check,omitempty"`
	DisableAdminRequests bool   `json:"disable_admin_requests,omitempty"`
	IgnoreCms            bool   `json:"ignore_cms,omitempty"`
	Reason               string `json:"reason,omitempty"`
	WithAutoHealing      bool   `json:"with_auto_healing,omitempty"`
	SSH                  string `json:"ssh,omitempty"`
}

type RebootKexecHostBody struct {
	Check                bool   `json:"check,omitempty"`
	DisableAdminRequests bool   `json:"disable_admin_requests,omitempty"`
	IgnoreCms            bool   `json:"ignore_cms,omitempty"`
	Reason               string `json:"reason,omitempty"`
	WithAutoHealing      bool   `json:"with_auto_healing,omitempty"`
}

type HandleFailureRequest struct {
	Check                bool   `json:"check,omitempty"`
	DisableAdminRequests bool   `json:"disable_admin_requests,omitempty"`
	IgnoreCms            bool   `json:"ignore_cms,omitempty"`
	Reason               string `json:"reason,omitempty"`
	WithAutoHealing      bool   `json:"with_auto_healing,omitempty"`
}

type RedeployHostRequest struct {
	HostID string
	Params RedeployHostParams
	Body   RedeployHostBody
}

type RebootHostRequest struct {
	HostID string
	Params RebootHostParams
	Body   RebootHostBody
}

type Health struct {
	CheckStatuses map[string]string `json:"check_statuses"`
}

type ProfileHostBody struct {
	Profile              string   `json:"profile,omitempty"`
	ProfileTags          []string `json:"profile_tags,omitempty"`
	Redeploy             bool     `json:"redeploy,omitempty"`
	Provisioner          string   `json:"provisioner,omitempty"`
	Config               string   `json:"config,omitempty"`
	DeployConfigPolicy   string   `json:"deploy_config_policy,omitempty"`
	DeployTags           []string `json:"deploy_tags,omitempty"`
	DeployNetwork        string   `json:"deploy_network,omitempty"`
	IgnoreCms            bool     `json:"ignore_cms,omitempty"`
	DisableAdminRequests bool     `json:"disable_admin_requests,omitempty"`
	Check                bool     `json:"check,omitempty"`
	WithAutoHealing      bool     `json:"with_auto_healing,omitempty"`
	Reason               string   `json:"reason,omitempty"`
}

type HostsStatus struct {
	Result     []*HostStatus `json:"result,omitempty"`
	Total      int           `json:"total"`
	NextCursor int           `json:"next_cursor"`
}

type GetHostsResponse struct {
	Hosts      []*pb.Host
	Total      int
	NextCursor int
}

type Location struct {
	Country             string `json:"country"`
	City                string `json:"city"`
	Rack                string `json:"rack"`
	Queue               string `json:"queue"`
	ShortDatacenterName string `json:"short_datacenter_name"`
}

type HealthChecksResp struct {
	Result []HealthCheck `json:"result,omitempty"`
	Total  int           `json:"total"`
}

type ProjectResp struct {
	DefaultHostRestrictions []string           `json:"default_host_restrictions"`
	HealingAutomation       *ProjectAutomation `json:"healing_automation"`
	DNSAutomation           *ProjectAutomation `json:"dns_automation"`
}

type ProjectAutomation struct {
	Enabled bool `json:"enabled"`
}

type HealthCheck struct {
	Fqdn        string      `json:"fqdn,omitempty"`
	ID          string      `json:"id,omitempty"`
	Metadata    interface{} `json:"metadata,omitempty"`
	Status      string      `json:"status,omitempty"`
	StatusMtime int         `json:"status_mtime,omitempty"`
	Timestamp   int         `json:"timestamp,omitempty"`
	Type        interface{} `json:"type,omitempty"`
}

type WalleFirmwareMetadata struct {
	Reason    string   `json:"reason,omitempty"`
	Status    string   `json:"status,omitempty"`
	Timestamp int      `json:"timestamp,omitempty"`
	Type      []string `json:"type,omitempty"`
}

type HostmanDistribMetadata struct {
	Reason    string `json:"reason,omitempty"`
	Timestamp int    `json:"timestamp,omitempty"`
}

type HostStatus struct {
	Inv          int       `json:"inv,omitempty"`
	Config       string    `json:"config,omitempty"`
	Name         string    `json:"name,omitempty"`
	State        string    `json:"state,omitempty"`
	Status       string    `json:"status,omitempty"`
	StatusAuthor string    `json:"status_author,omitempty"`
	Location     *Location `json:"location,omitempty"`
	UUID         string    `json:"uuid,omitempty"`
	Project      string    `json:"project,omitempty"`
	Restrictions []string  `json:"restrictions"`
	Health       Health    `json:"health"`
	Tags         []string  `json:"tags"`
	Task         *Task     `json:"task,omitempty"`
	Ticket       string    `json:"ticket"`
}

type Task struct {
	Status  string `json:"status"`
	Message string `json:"status_message"`
}

type RedeployHostParams struct {
	IgnoreMaintenance bool
}

type RebootHostParams struct {
	IgnoreMaintenance bool
}

func (params *RebootHostParams) Format() map[string]string {
	return map[string]string{
		"ignore_maintenance": strconv.FormatBool(params.IgnoreMaintenance),
	}
}

func (params *RedeployHostParams) Format() map[string]string {
	return map[string]string{
		"ignore_maintenance": strconv.FormatBool(params.IgnoreMaintenance),
	}
}

type GetHostsRequestParams struct {
	Name                       string
	State                      string
	Status                     string
	Health                     string
	HealthIn                   string
	HealthNin                  string
	Project                    string
	LocationCity               string
	LocationCountry            string
	Tags                       string
	Provisioner                string
	Config                     string
	DeployConfigPolicy         string
	Restrictions               string
	RestrictionsIn             string
	RestrictionsNin            string
	TaskOwner                  string
	PhysicalLocation           string
	Switch                     string
	Port                       string
	ResolveDeployConfiguration string
	ResolveTags                string
	ScenarioID                 string
	Fields                     []string
	Cursor                     int
	Offset                     string
	Limit                      int64
}

func (params *GetHostsRequestParams) Format() map[string]string {
	return map[string]string{
		"name":                         params.Name,
		"state":                        params.State,
		"status":                       params.Status,
		"health":                       params.Health,
		"health__in":                   params.HealthIn,
		"health__nin":                  params.HealthNin,
		"project":                      params.Project,
		"tags":                         params.Tags,
		"task_owner":                   params.TaskOwner,
		"provisioner":                  params.Provisioner,
		"config":                       params.Config,
		"deploy_config_policy":         params.DeployConfigPolicy,
		"restrictions":                 params.Restrictions,
		"physical_location":            params.PhysicalLocation,
		"switch":                       params.Switch,
		"port":                         params.Port,
		"scenario_id":                  params.ScenarioID,
		"resolve_deploy_configuration": params.ResolveDeployConfiguration,
		"resolve_tags":                 params.ResolveTags,
		"fields":                       strings.Join(params.Fields[:], ","),
		"cursor":                       fmt.Sprintf("%d", params.Cursor),
		"offset":                       params.Offset,
		"limit":                        strconv.FormatInt(params.Limit, 10),
	}
}
