package yp

import (
	"context"
	"errors"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
	"a.yandex-team.ru/yp/go/yp"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

var ErrNotFetched = errors.New("nothing fetched")

type YPCluster string

// bug: man cluster disabled

const (
	YPClusterSAS     = "sas"
	YPClusterVLA     = "vla"
	YPClusterIVA     = "iva"
	YPClusterMYT     = "myt"
	YPClusterSASTEST = "sas-test"
	YPClusterMANPRE  = "man-pre"
	YPClusterXDC     = "xdc"
	//YPClusterMAN     = "man"
)

//var YpClusters = []YPCluster{YPClusterSAS, YPClusterMAN, YPClusterVLA, YPClusterIVA, YPClusterMYT, YPClusterXDC, YPClusterSASTEST, YPClusterMANPRE}

var YpClusters = []YPCluster{YPClusterSAS, YPClusterVLA, YPClusterIVA, YPClusterMYT, YPClusterXDC, YPClusterSASTEST, YPClusterMANPRE}

type ClientPool map[YPCluster]*client

// GetYpClientsPool is init yp client's pool function (for all yp clusters)
func GetYpClientsPool(logger log.Logger, clusters []YPCluster, token string) (map[YPCluster]*client, error) {
	pool := make(ClientPool)
	for _, c := range clusters {
		config := ClientConfig{Cluster: c, Insecure: false, Token: token}
		cclient, err := NewClient(logger, &config)
		if err != nil {
			return nil, err
		}
		pool[c] = cclient
	}
	return pool, nil
}

// ClientConfig is yp client config
type ClientConfig struct {
	Cluster  YPCluster
	Address  string
	Insecure bool
	Token    string
}

// rawResponse is mockable yp selectobject response interfaces
type rawResponse interface {
	Next() bool
	Fill(...interface{}) error
	Count() int
}

// rawClient is mockable yp client interface
type rawClient interface {
	SelectObjects(context.Context, yp.SelectObjectsRequest) (rawResponse, error)
}

// rawClientWrapper wraps *yp.Client for type erasure
type rawClientWrapper struct {
	c *yp.Client
}

// SelectObjects function is a wrapper for yp.Client SelectObjects method
func (r *rawClientWrapper) SelectObjects(ctx context.Context, request yp.SelectObjectsRequest) (rawResponse, error) {
	return r.c.SelectObjects(ctx, request)
}

// client compatible with mock
type client struct {
	rawClient rawClient
}

// Client interfaces
type Client interface {
	GetPods(ctx context.Context) ([]*Pod, error)
	GetPodSets(ctx context.Context) ([]*PodSet, error)
	GetBudgets(ctx context.Context) ([]*Budget, error)
}

// NewClient function returns new yp client
func NewClient(logger log.Logger, config *ClientConfig) (*client, error) {
	err := config.validate()
	if err != nil {
		return nil, err
	}
	options := []yp.Option{
		yp.WithLogger(logger),
	}
	options = append(options, yp.WithAuthToken(config.Token))
	if config.Address != "" {
		options = append(options, yp.WithAddress(config.Address))
	}
	if config.Insecure {
		options = append(options, yp.WithInsecure())
	}
	cli, err := yp.NewClient(string(config.Cluster), options...)
	if err != nil {
		return nil, err
	}
	return &client{rawClient: &rawClientWrapper{cli}}, nil
}

// GetPods function return pods within podset and nanny flags information
func (c *client) GetPods(ctx context.Context) ([]*Pod, error) {
	res, err := c.rawClient.SelectObjects(ctx, yp.SelectObjectsRequest{
		ObjectType: yp.ObjectTypePod,
		Selectors:  []string{"/status/scheduling/node_id", "/meta/pod_set_id", "/labels/nanny_service_id"},
		Format:     yp.PayloadFormatYson,
	})
	if err != nil {
		return nil, err
	}
	if res.Count() == 0 {
		return nil, ErrNotFetched
	}
	var pods []*Pod
	for res.Next() {
		pod := &Pod{}
		if err := res.Fill(&pod.NodeID, &pod.PodsetID, &pod.NannyID); err != nil {
			return nil, err
		}
		pods = append(pods, pod)
	}
	return pods, nil
}

// Pod is an yp pod query response structure
type Pod struct {
	NodeID   string
	PodsetID string
	NannyID  string
}

// GetPodSets function returns podsets within pdb flag
func (c *client) GetPodSets(ctx context.Context) ([]*PodSet, error) {
	res, err := c.rawClient.SelectObjects(ctx, yp.SelectObjectsRequest{
		ObjectType: yp.ObjectTypePodSet,
		Selectors:  []string{"/meta/id", "/spec/pod_disruption_budget_id"},
		Filter:     "",
		Format:     yp.PayloadFormatYson,
	})
	if err != nil {
		return nil, err
	}
	if res.Count() == 0 {
		return nil, ErrNotFetched
	}
	var podsets []*PodSet
	for res.Next() {
		podset := &PodSet{}
		if err := res.Fill(&podset.MetaID, &podset.BudgetID); err != nil {
			return nil, err
		}
		podsets = append(podsets, podset)
	}
	return podsets, nil
}

// GetBudgets function returns pdb budgets
func (c *client) GetBudgets(ctx context.Context) ([]*Budget, error) {
	res, err := c.rawClient.SelectObjects(ctx, yp.SelectObjectsRequest{
		ObjectType: yp.ObjectTypePodDisruptionBudget,
		Selectors:  []string{"/meta/id", "/status/allowed_pod_disruptions"},
		Format:     yp.PayloadFormatYson,
	})
	if err != nil {
		return nil, err
	}
	if res.Count() == 0 {
		return nil, ErrNotFetched
	}
	var budgets []*Budget
	for res.Next() {
		budget := &Budget{}
		if err := res.Fill(&budget.MetaID, &budget.NumberAllowed); err != nil {
			return nil, err
		}
		budgets = append(budgets, budget)
	}
	return budgets, nil
}

// PodSet is an yp podset query response structure
type PodSet struct {
	MetaID   string
	BudgetID string
}

// Budget is an yp pod desruption budget query response
type Budget struct {
	MetaID        string
	NumberAllowed int
}

// validate yp client config
func (c *ClientConfig) validate() error {
	if c.Token == "" {
		return errors.New("token not set in yp client config")
	}
	if c.Address == "" && c.Cluster == "" {
		return errors.New("both address and cluster not set in yp client config")
	}
	return nil
}

// GetPodsetPdbs map pod disruption budget policies on podsets
func GetPodsetPdbs(podsets []*PodSet, budgets []*Budget) map[string]int {
	// make map for budget map
	bm := make(map[string]int)

	// make map for pod disruption budgets map
	pdbs := make(map[string]int)

	for _, b := range budgets {
		bm[b.MetaID] = b.NumberAllowed
	}
	for _, ps := range podsets {
		if budget, ok := bm[ps.BudgetID]; ok {
			pdbs[ps.MetaID] = budget
		} else {
			// not all yp podset have bdb limit, so let's suppose them unlimited
			pdbs[ps.MetaID] = 999
		}
	}
	return pdbs
}

// CompileNodes return nodes data from pods and pdbs info
func CompileNodes(pods []*Pod, pdbs map[string]int) map[string]*pb.YpNode {
	nodes := make(map[string]*pb.YpNode)
	for _, p := range pods {
		// if node without podsets or non yp node, so than pdb not applicable by default
		minbudget := 999
		if b, ok := pdbs[p.PodsetID]; ok {
			minbudget = b
		}
		if _, ok := nodes[p.NodeID]; !ok {
			nodes[p.NodeID] = &pb.YpNode{MetaId: p.NodeID, BudgetMinimum: int32(minbudget), NannyServices: []string{p.NannyID}}
			continue
		}
		node := nodes[p.NodeID]
		if node.BudgetMinimum > int32(minbudget) {
			node.BudgetMinimum = int32(minbudget)
		}
		// check duplicates for nanny services in node information
		if slices.ContainsString(node.NannyServices, p.NannyID) {
			node.NannyServices = append(node.NannyServices, p.NannyID)
		}
		nodes[p.NodeID] = node

	}
	return nodes
}
