package yp

import (
	"a.yandex-team.ru/yp/go/yp"
	"context"
	"errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"reflect"
	"testing"

	pb "a.yandex-team.ru/infra/maxwell/go/proto"
)

func TestNewClient(t *testing.T) {
	cfg := ClientConfig{
		Cluster:  "",
		Address:  "",
		Insecure: false,
		Token:    "",
	}
	exp, err := NewClient(nil, &cfg)
	require.Error(t, err)
	require.Equal(t, exp, (*client)(nil))
}

type rspMock struct {
	t    *testing.T
	i    int
	objs [][]interface{}
}

func (p *rspMock) Next() bool {
	p.i++
	return p.i <= len(p.objs)
}

func (p *rspMock) Fill(o ...interface{}) error {
	objIdx := p.i - 1
	if objIdx < 0 || objIdx >= len(p.objs) {
		return errors.New("no nex row")
	}
	for i := range o {
		lval := reflect.ValueOf(o[i])
		rval := reflect.ValueOf(p.objs[objIdx][i])
		lval.Elem().Set(rval)
	}
	return nil
}

func (p *rspMock) Count() int {
	return len(p.objs)
}

type wrappedMock struct {
	t *testing.T
	r *rspMock
}

func (p *wrappedMock) SelectObjects(ctx context.Context, request yp.SelectObjectsRequest) (rawResponse, error) {
	assert.NotEqual(p.t, request.Selectors, "")
	return p.r, nil
}

func TestClient_GetPods(t *testing.T) {
	m := &wrappedMock{t: t, r: &rspMock{}}
	c := client{rawClient: m}
	r, err := c.GetPods(context.TODO())
	assert.Error(t, err)
	assert.Nil(t, r)
	m.r.objs = [][]interface{}{{"node1", "podset1", "nanny1"}, {"node2", "podset2", "nanny2"}}
	r, err = c.GetPods(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, 2, len(r))
	assert.Equal(t, &Pod{
		NodeID:   "node2",
		PodsetID: "podset2",
		NannyID:  "nanny2",
	}, r[1])
}

func TestClient_GetPodSets(t *testing.T) {
	m := &wrappedMock{t: t, r: &rspMock{}}
	c := client{rawClient: m}
	r, err := c.GetPodSets(context.TODO())
	assert.Error(t, err)
	assert.Nil(t, r)
	m.r.objs = [][]interface{}{{"podset1", "budgetid1"}, {"podset2", "budgetid2"}}
	r, err = c.GetPodSets(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, 2, len(r))
	assert.Equal(t, &PodSet{
		MetaID:   "podset1",
		BudgetID: "budgetid1",
	}, r[0])
}

func TestClient_GetBudgets(t *testing.T) {
	m := &wrappedMock{t: t, r: &rspMock{}}
	c := client{rawClient: m}
	r, err := c.GetBudgets(context.TODO())
	assert.Error(t, err)
	assert.Nil(t, r)
	m.r.objs = [][]interface{}{{"budgetid1", 1}, {"budgetid2", 999}}
	r, err = c.GetBudgets(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, 2, len(r))
	assert.Equal(t, &Budget{
		MetaID:        "budgetid1",
		NumberAllowed: 1,
	}, r[0])
}

func TestCompileNodes(t *testing.T) {
	assert.Equal(t, CompileNodes([]*Pod{&Pod{NodeID: "node1", PodsetID: "podset1", NannyID: "nanny1"}}, map[string]int{"podset1": 12}),
		map[string]*pb.YpNode{"node1": &pb.YpNode{MetaId: "node1", BudgetMinimum: 12, NannyServices: []string{"nanny1"}}})

	// check rule for non limited by pdb services
	assert.Equal(t, CompileNodes([]*Pod{&Pod{NodeID: "node1", PodsetID: "podset1", NannyID: "nanny1"}}, map[string]int{"otherpodset": 1}),
		map[string]*pb.YpNode{"node1": &pb.YpNode{MetaId: "node1", BudgetMinimum: 999, NannyServices: []string{"nanny1"}}})

}
