from __future__ import unicode_literals
import logging
import logging.handlers
import os

import flask
import gevent
import raven.conf
import raven.handlers.logging
import raven.transport.threaded

import yp.client
from sepelib.core import config
from sepelib.util import log as logutil
from infra.swatlib import cmdutil
from infra.swatlib import metrics
from infra.swatlib import logutil as nlogutil
from infra.ytexclusiveservice import yt_exclusiveservice
from infra.mc_rsc.src import circuit_breaker
from infra.mc_rsc.src import consts
from infra.mc_rsc.src import gc
from infra.mc_rsc.src import manager
from infra.mc_rsc.src import metrics_collector
from infra.mc_rsc.src import rate_limiter
from infra.mc_rsc.src import reflector
from infra.mc_rsc.src import runner
from infra.mc_rsc.src import storage
from infra.mc_rsc.src import yputil
from infra.swatlib import webserver
from infra.mc_rsc.src.controller import controller
from infra.mc_rsc.src.lib import yp_client
from infra.mc_rsc.src.lib import loaders


PRODUCTION_XDC = 'xdc'


class Application(object):
    name = 'mc_rsc'

    def __init__(self, instance_id):
        self.instance_id = instance_id
        self.log = logging.getLogger(self.name)

    @staticmethod
    def setup_environment():
        # Patch requests connection pool to use gevent queue
        from requests.packages.urllib3.connectionpool import ConnectionPool
        from gevent.queue import LifoQueue

        ConnectionPool.QueueCls = LifoQueue
        # Disable requests spamming about:
        # Connection pool is full, discarding connection
        # This is the way we use alemate-http to avoid broken connections
        # There is nothing we can do about it, so simply mute
        logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.ERROR)
        ylock_log_config = config.get_value('coord.log')
        ylock_log_handler = logging.handlers.RotatingFileHandler(**ylock_log_config)
        ylock_logger = logging.getLogger('Yt')
        ylock_logger.addHandler(ylock_log_handler)
        ylock_logger.setLevel(logging.DEBUG)

    @classmethod
    def setup_yp_client_logging(cls):
        logcfg = config.get_value('yp_client_log', {})
        if not logcfg:
            return
        filepath = logcfg.get('filepath')
        if filepath:
            logcfg['filepath'] = nlogutil.add_logfile_prefix(filepath, cls.name)
        handler = logutil.create_handler_from_config(logcfg)
        logger = logutil.setup_logging_to_file(handler,
                                               redirect_stderr=False,
                                               redirect_stdout=False,
                                               logger_name='YpClient')
        cmdutil.setup_exception_logging()
        environ_level = os.environ.get('YP_LOG_LEVEL')
        conf_level = logcfg.get('loglevel')
        level = environ_level or conf_level or 'INFO'
        loglevel = logging.getLevelName(level.upper())
        logger.setLevel(loglevel)
        logger.propagate = False

    @staticmethod
    def setup_logging(sentry_dsn=None):
        if not sentry_dsn:
            return
        c = raven.Client(
            dsn=sentry_dsn,
            transport=raven.transport.threaded.ThreadedHTTPTransport
        )
        handler = raven.handlers.logging.SentryHandler(c, level=logging.ERROR)
        raven.conf.setup_logging(handler)

    def _run_webserver(self, reg):
        # 1.
        # `app` and `self.web` have to be created and run in the same native thread
        # due to their use of hub-local events.
        # 2.
        # We run webserver in a separate greenlet to make sure there are at *least two*
        # of them in this native thread. This is to avoid random "this operation would block forever"
        # errors, see https://github.com/gevent/gevent/issues/890 for more details.
        # 3.
        # We have to make sure that LocalWebServer does not use logging module, as it will lead it to
        # an attempt to acquire logging's lock from the main gevent hub and failure.
        app = flask.Flask(self.name)
        self.web = webserver.LocalWebServer(cfg=config.get(), app=app, version='stable', metrics_registry=reg)
        gevent.spawn(self.web.run).get()

    def run(self):
        self.setup_environment()

        sentry_dsn = config.get_value('sentry.dsn', default=None)
        self.setup_logging(sentry_dsn=sentry_dsn)
        self.setup_yp_client_logging()

        reg = metrics.Registry()

        # Instantiate yp clients.
        yp_clients = {}
        clusters = set()
        for c in config.get_value('yp.clusters'):
            cluster = c['cluster']
            clusters.add(cluster)
            address = c['address']
            token = c.get('token')
            if not token:
                c['token'] = os.environ.get('YP_TOKEN')
            threadpool_max_size = c.get('threadpool_max_size', 10)
            base = yp.client.YpClient(address=address, config=c)
            stub = base.create_grpc_object_stub()
            payload_format = c.get('payload_format', 'yson')
            loader = loaders.make_loader(payload_format)
            yp_clients[cluster] = yp_client.YpClient(stub=stub,
                                                     loader=loader,
                                                     threadpool_max_size=threadpool_max_size)

        # Instantiate storages, reflectors and gcs.
        xdc = config.get_value('yp.xdc')
        deploy_engine = config.get_value('deploy_engine')
        select_ids_batch_size = config.get_value('reflector.select_ids_batch_size')
        event_count_limit = config.get_value('reflector.watch_event_count_limit', 100)
        get_objects_batch_size = config.get_value('reflector.get_objects_batch_size')
        reflector_sleep_secs = config.get_value('reflector.sleep_secs')
        watch_time_limit_secs = config.get_value('reflector.watch_time_limit_secs')
        select_threads_count = config.get_value('reflector.select_threads_count')
        mc_rs_match_dict = config.get_value('reflector.mc_rs_match_labels',
                                            default={})
        gc_sleep_secs = config.get_value('gc.sleep_secs')
        mc_rs_match_dict.pop('deploy_engine', None)
        pod_match_dict = config.get_value('reflector.pod_match_labels')
        pod_match_dict['deploy_engine'] = deploy_engine

        mc_rs_deploy_engine_filter = (
            '[/labels/deploy_engine] = "{}" OR '
            '[/labels/deploy_engine] = null'
        ).format(deploy_engine)
        if mc_rs_match_dict:
            mc_rs_filter = '({}) AND ({})'.format(
                mc_rs_deploy_engine_filter,
                yputil.make_filter_from_dict(mc_rs_match_dict)
            )
        else:
            mc_rs_filter = mc_rs_deploy_engine_filter
        pod_filter = yputil.make_filter_from_dict(pod_match_dict)

        mc_rs_storage = storage.ClusterStorage()
        mc_rs_reflector = reflector.make_mc_rs_reflector(
            deploy_engine=deploy_engine,
            cluster=xdc,
            obj_filter=mc_rs_filter,
            selectors=consts.DEFAULT_OBJECT_SELECTORS,
            watch_selectors=consts.DEFAULT_OBJECT_SELECTORS,
            use_watches=False,
            fetch_timestamps=False,
            storage=mc_rs_storage,
            client=yp_clients[xdc],
            select_ids_batch_size=select_ids_batch_size,
            event_count_limit=event_count_limit,
            get_objects_batch_size=get_objects_batch_size,
            watch_time_limit_secs=watch_time_limit_secs,
            sleep_secs=reflector_sleep_secs,
            select_threads_count=select_threads_count,
            metrics_registry=reg
        )
        relation_storage = storage.RelationClusterStorage()
        relation_reflector = reflector.Reflector(
            name='relation',
            cluster=cluster,
            obj_type=yp.data_model.OT_RELATION,
            obj_class=yp.data_model.TRelation,
            obj_filter=None,
            selectors=consts.RELATION_SELECTORS,
            watch_selectors=consts.DEFAULT_OBJECT_SELECTORS,
            use_watches=False,
            fetch_timestamps=False,
            storage=relation_storage,
            client=yp_clients[xdc],
            select_ids_batch_size=select_ids_batch_size,
            event_count_limit=event_count_limit,
            get_objects_batch_size=get_objects_batch_size,
            watch_time_limit_secs=watch_time_limit_secs,
            sleep_secs=reflector_sleep_secs,
            select_threads_count=select_threads_count,
            metrics_registry=reg,
        )
        pod_multi_cluster_storage = storage.PodMultiClusterStorage()
        ps_multi_cluster_storage = storage.MultiClusterStorage()
        ps_gcs = []

        use_watches = config.get_value('watches.use_for_pods', False)
        watches_dry_run = config.get_value('watches.dry_run', False)
        pod_reflectors = {}
        ps_reflectors = {}
        for cluster in clusters:
            client = yp_clients[cluster]
            pod_storage = storage.PodClusterStorage()
            if watches_dry_run:
                pod_storage_full_sync = storage.PodClusterStorage()
                pod_multi_cluster_storage.add_storage(pod_storage_full_sync, cluster)
            else:
                pod_storage_full_sync = None
                pod_multi_cluster_storage.add_storage(pod_storage, cluster)
            ps_storage = storage.ClusterStorage()
            ps_multi_cluster_storage.add_storage(ps_storage, cluster)
            if cluster.lower() == PRODUCTION_XDC:
                # Do not strart pod/pod_set reflectors and pod_set gc in xdc
                # cluster, because there are no pods/pod_sets.
                continue
            pod_reflector = reflector.Reflector(
                name='pod',
                cluster=cluster,
                obj_type=yp.data_model.OT_POD,
                obj_class=yp.data_model.TPod,
                obj_filter=pod_filter,
                selectors=client.loader.get_selectors_by_object_type(yp.data_model.OT_POD),
                watch_selectors=consts.POD_WATCH_SELECTORS,
                use_watches=use_watches,
                fetch_timestamps=True,
                storage=pod_storage,
                client=client,
                select_ids_batch_size=select_ids_batch_size,
                event_count_limit=event_count_limit,
                get_objects_batch_size=get_objects_batch_size,
                watch_time_limit_secs=watch_time_limit_secs,
                sleep_secs=reflector_sleep_secs,
                select_threads_count=select_threads_count,
                metrics_registry=reg,
                full_sync_storage=pod_storage_full_sync,
            )
            pod_reflectors[cluster] = pod_reflector
            ps_reflector = reflector.Reflector(
                name='ps',
                cluster=cluster,
                obj_type=yp.data_model.OT_POD_SET,
                obj_class=yp.data_model.TPodSet,
                obj_filter=pod_filter,
                selectors=client.loader.get_selectors_by_object_type(yp.data_model.OT_POD_SET),
                watch_selectors=consts.DEFAULT_OBJECT_SELECTORS,
                use_watches=False,
                fetch_timestamps=False,
                storage=ps_storage,
                client=client,
                select_ids_batch_size=select_ids_batch_size,
                event_count_limit=event_count_limit,
                get_objects_batch_size=get_objects_batch_size,
                watch_time_limit_secs=watch_time_limit_secs,
                sleep_secs=reflector_sleep_secs,
                select_threads_count=select_threads_count,
                metrics_registry=reg
            )
            ps_reflectors[cluster] = ps_reflector

            ps_gc = gc.make_ps_gc(
                deploy_engine=deploy_engine,
                name='pod_set_gc',
                xdc_client=yp_clients[xdc],
                cluster=cluster,
                client=client,
                mc_rs_storage=mc_rs_storage,
                ps_storage=ps_storage,
                sleep_secs=gc_sleep_secs,
                metrics_registry=reg,
            )
            ps_gcs.append(ps_gc)

        # Instantiate ctl.
        root_users = config.get_value('yp.root_users', [])
        pod_match_dict['deploy_engine'] = deploy_engine
        match_labels = yputil.make_labels_from_dict(pod_match_dict)
        max_pods_to_process = config.get_value('controller.max_pods_to_process', 100)
        update_pods_batch_size = config.get_value('controller.update_pods_batch_size', 10)
        replace_pods_batch_size = config.get_value('controller.replace_pods_batch_size', 5)
        reallocate_pods_batch_size = config.get_value('controller.reallocate_pods_batch_size', 10)

        max_tries = config.get_value(
            'circuit_breaker.max_tries',
            consts.DEFAULT_CIRCUIT_BREAKER_MAX_TRIES
        )
        is_circuit_breaker_enabled = config.get_value(
            'circuit_breaker.enabled',
            True
        )
        breaker = circuit_breaker.CircuitBreaker(
            max_tries=max_tries,
            is_enabled=is_circuit_breaker_enabled
        )

        is_rate_limiter_enabled = config.get_value('rate_limiter.enabled',
                                                   True)
        rate_limiter_delay_secs = config.get_value(
            'rate_limiter.delay_secs',
            consts.DEFAULT_RATE_LIMITER_DELAY_SECS
        )
        limiter = rate_limiter.RateLimiter(is_enabled=is_rate_limiter_enabled,
                                           delay_secs=rate_limiter_delay_secs)
        client = yp_clients[xdc]
        stop_process_mc_rs_with_unrelated_children = config.get_value(
            'controller.stop_process_mc_rs_with_unrelated_children', False
        )
        ctl = controller.make_controller(
            deploy_engine=deploy_engine,
            root_users=root_users,
            yp_clients=yp_clients,
            match_labels=match_labels,
            max_pods_to_process=max_pods_to_process,
            update_pods_batch_size=update_pods_batch_size,
            replace_pods_batch_size=replace_pods_batch_size,
            reallocate_pods_batch_size=reallocate_pods_batch_size,
            clusters=clusters,
            mc_rs_cluster=xdc,
            pod_storage=pod_multi_cluster_storage,
            ps_storage=ps_multi_cluster_storage,
            relation_storage=relation_storage,
            circuit_breaker=breaker,
            rate_limiter=limiter,
            stop_process_mc_rs_with_unrelated_children=stop_process_mc_rs_with_unrelated_children
        )

        # Instantiate ctl runner.
        threads_count = config.get_value('runner.threads_count')
        relation_sync_delay = config.get_value('runner.relation_sync_delay', 5 * 60)
        allowed_mc_rs_ids = set(config.get_value('runner.allowed_mc_rs_ids', []))
        ignored_mc_rs_ids = set(config.get_value('runner.ignored_mc_rs_ids', []))
        assert not allowed_mc_rs_ids or not ignored_mc_rs_ids

        collector = metrics_collector.MetricsCollector(metrics_registry=reg,
                                                       cluster=xdc)
        ctl_runner = runner.Runner(ctl=ctl,
                                   deploy_engine=deploy_engine,
                                   xdc=xdc,
                                   yp_clients=yp_clients,
                                   mc_rs_storage=mc_rs_storage,
                                   circuit_breaker=breaker,
                                   pod_reflectors=pod_reflectors,
                                   ps_reflectors=ps_reflectors,
                                   mc_rs_reflector=mc_rs_reflector,
                                   relation_reflector=relation_reflector,
                                   threads_count=threads_count,
                                   relation_sync_delay=relation_sync_delay,
                                   allowed_mc_rs_ids=allowed_mc_rs_ids,
                                   ignored_mc_rs_ids=ignored_mc_rs_ids,
                                   metrics_registry=reg,
                                   metrics_collector=collector)

        # Instantiate manager.
        mngr = manager.Manager(ps_gcs=ps_gcs, ctl_runner=ctl_runner)
        yt_manager_cfg = config.get_value('coord.manager')
        lock_name = config.get_value('coord.lock.name')
        expiration_timeout = config.get_value('coord.lock.expiration_timeout')
        acquire_timeout = config.get_value('coord.lock.acquire_timeout')
        # self.manager = mngr
        self.manager = yt_exclusiveservice.YtExclusiveService(
            cfg=yt_manager_cfg,
            name=lock_name,
            runnable=mngr,
            lock_timeout=expiration_timeout,
            acquire_timeout=acquire_timeout,
            metrics_registry=reg)

        self.pool = gevent.threadpool.ThreadPool(1)
        self.pool.spawn(self._run_webserver, reg=reg)

        # Let's start here.
        self.log.info('starting service...')

        self.manager.start()
        self.manager.wait()

    def teardown_environment(self):
        pass

    def stop(self):
        """
        Gracefully stop application.
        Can block for a long time or throw exception, be ready.
        """
        self.log.info('stopping service...')
        self.manager.stop()
        self.pool.stop()
        self.teardown_environment()
        self.log.info('Done.')
