from __future__ import unicode_literals
import itertools
import logging
import time


import yp.data_model
import yp.common
from sepelib.core import config
from infra.swatlib import randomutil
from infra.swatlib.gevent import geventutil as gutil
from infra.mc_rsc.src import consts
from infra.mc_rsc.src import deploy_policies
from infra.mc_rsc.src import disruption_policies
from infra.mc_rsc.src import podutil
from infra.mc_rsc.src import state
from infra.mc_rsc.src import status_maker
from infra.mc_rsc.src import yp_client_adapter
from infra.mc_rsc.src import yputil
from infra.mc_rsc.src.pod_actions import PodActionKind
from infra.mc_rsc.src.controller import cluster_task
from infra.mc_rsc.src.controller import exceptions
from infra.mc_rsc.src.controller import validation
from infra.mc_rsc.src.controller import ctlutil


class ClustersInfo(object):
    STR_TPL = ("ok_spec_clusters: {ok_spec_clusters}, "
               "failed_spec_clusters: {failed_spec_clusters} "
               "not_in_spec_clusters: {not_in_spec_clusters} "
               "disabled_clusters: {disabled_clusters} ")

    def __init__(self):
        self.ok_spec_clusters = set()
        self.failed_spec_clusters = set()
        self.not_in_spec_clusters = set()
        self.disabled_clusters = set()

    def __str__(self):
        return self.STR_TPL.format(
            ok_spec_clusters=self.ok_spec_clusters,
            failed_spec_clusters=self.failed_spec_clusters,
            not_in_spec_clusters=self.not_in_spec_clusters,
            disabled_clusters=self.disabled_clusters,
        )


class ProcessStatistic(object):
    def __init__(self):
        self.unrelated_children_found = False
        self.is_mc_rs_ready = False
        self.updated_pod_sets = 0
        self.removed_pod_sets = 0
        self.eviction_acknowledged_pods = 0
        self.maintenance_acknowledged_pods = 0
        self.created_pods = 0
        self.removed_pods = 0
        self.reallocated_inplace_pods = 0
        self.replaced_pods = 0
        self.updated_pods = 0
        self.eviction_requested_pods = 0
        self.set_target_state_removed_pods = 0
        self.set_target_state_active_pods = 0


class Controller(object):

    ACKNOWLEDGE_MSG = 'Acknowledged by RSC'
    EVICTION_REQUEST_MSG = 'Requested by RSC'
    DELEGATE_REMOVING_MSG = 'Delegate pod removing'

    def __init__(self, yp_clients, mc_rs_client,
                 default_acl,
                 match_labels,
                 max_pods_to_process,
                 update_pods_batch_size,
                 replace_pods_batch_size,
                 reallocate_pods_batch_size,
                 clusters,
                 pod_storage, ps_storage, relation_storage,
                 mc_rs_status_maker,
                 circuit_breaker,
                 rate_limiter,
                 stop_process_mc_rs_with_unrelated_children=False):
        super(Controller, self).__init__()
        self.yp_clients = yp_clients
        self.mc_rs_client = mc_rs_client
        self.default_acl = default_acl
        self.match_labels = match_labels
        self.max_pods_to_process = max_pods_to_process
        self.update_pods_batch_size = update_pods_batch_size
        self.replace_pods_batch_size = replace_pods_batch_size
        self.reallocate_pods_batch_size = reallocate_pods_batch_size
        self.clusters = clusters
        self.mc_rs_status_maker = mc_rs_status_maker
        self.pod_storage = pod_storage
        self.ps_storage = ps_storage
        self.relation_storage = relation_storage
        self.circuit_breaker = circuit_breaker
        self.rate_limiter = rate_limiter
        self.stop_process_mc_rs_with_unrelated_children = stop_process_mc_rs_with_unrelated_children
        self._init_deploy_policies()

    def _init_deploy_policies(self):
        self.deploy_policies = {
            'default': deploy_policies.DeployPolicy(
                pod_storage=self.pod_storage,
                pod_disruption_policy=disruption_policies.RollingUpdateDisruptionPolicy(
                    pod_storage=self.pod_storage
                )
            ),
            'remove_delegation': deploy_policies.DeployPolicy(
                pod_storage=self.pod_storage,
                pod_disruption_policy=disruption_policies.RemoveDelegationDisruptionPolicy(
                    pod_storage=self.pod_storage
                )
            ),
            'remove_delegation_parallel_reallocation': deploy_policies.DeployPolicy(
                pod_storage=self.pod_storage,
                pod_disruption_policy=disruption_policies.RemoveDelegationDisruptionPolicy(
                    pod_storage=self.pod_storage,
                    parallel_reallocation=True
                )
            )
        }
        self.default_policy = self.deploy_policies['default']

    def _make_clusters_info(self, mc_rs, failed_clusters):
        rv = ClustersInfo()
        disabled_clusters = mc_rs.disabled_clusters()
        for c in self.clusters:
            is_ok = c not in failed_clusters
            has_cluster = mc_rs.has_cluster(c)
            if c in disabled_clusters:
                rv.disabled_clusters.add(c)
            elif is_ok and has_cluster:
                rv.ok_spec_clusters.add(c)
            elif is_ok and not has_cluster:
                rv.not_in_spec_clusters.add(c)
            elif not is_ok and has_cluster:
                rv.failed_spec_clusters.add(c)
        return rv

    def _make_current_state(self, mc_rs, ok_spec_clusters):
        rv = state.MultiClusterCurrentState(mc_rs.spec.deployment_strategy.max_tolerable_downtime_seconds)
        for cluster in ok_spec_clusters:
            ps_id = mc_rs.make_ps_id()
            for p in self.pod_storage.list_by_ps_id(ps_id, cluster):
                rv.add(p, cluster)
        return rv

    def _make_cluster_task(self, task, cluster, mc_rs_id,
                           check_rate_limiter, logger,
                           task_args=None, task_kwargs=None):
        return cluster_task.ClusterTask(
            task=task,
            task_args=task_args,
            task_kwargs=task_kwargs,
            cluster=cluster,
            mc_rs_id=mc_rs_id,
            rate_limiter=self.rate_limiter,
            circuit_breaker=self.circuit_breaker,
            check_rate_limiter=check_rate_limiter,
            logger=logger
        )

    def get_yp_client(self, cluster):
        return self.yp_clients[cluster]

    def update_pod_set_in_cluster(self, ps_id, template, cluster, logger, stat):
        logger.info('[%s] updating pod_set: %s', cluster, ps_id)
        client = self.get_yp_client(cluster)
        client.update_pod_set(ps_id=ps_id, template=template)
        stat.updated_pod_sets += 1
        logger.info('[%s] updated pod_set: %s', cluster, ps_id)

    def remove_pod_set_in_cluster(self, ps, mc_rs, cluster, logger, stat):
        logger.info('[%s] removing pod_set: %s', cluster, ps.meta.id)
        client = self.get_yp_client(cluster)
        tid, _ = client.start_transaction()
        client.remove_pod_set(pod_set_id=ps.meta.id, transaction_id=tid)
        if config.get_value('controller.remove_deploy_child_on_pod_set_removing', False):
            self.mc_rs_client.remove_deploy_child(mc_rs_id=mc_rs.meta.id, child=ps.meta.fqid)
        client.commit_transaction(tid)
        if config.get_value('controller.remove_deploy_child_on_pod_set_removing', False):
            rels = self.relation_storage.find(from_fqid=mc_rs.meta.fqid,
                                              to_fqid=ps.meta.fqid)
            for r in gutil.gevent_idle_iter(rels):
                self.relation_storage.remove(r.meta.id)
        stat.removed_pod_sets += 1
        logger.info('[%s] removed pod_set: %s', cluster, ps.meta.id)

    def get_max_pods_to_process(self, mc_rs):
        update_portion = mc_rs.get_deploy_speed().update_portion
        if not update_portion:
            return self.max_pods_to_process
        return min(update_portion, self.max_pods_to_process)

    # TODO: split this methods into 2: create_pods_in_cluster and
    # create_pods_with_pod_set_in_cluster and move pod_set validation to
    # create_pods method.
    def create_pods_in_cluster(self, mc_rs, pods, cluster, logger, stat):
        pods = pods[:self.get_max_pods_to_process(mc_rs)]
        logger.info('[%s] creating %d pods', cluster, len(pods))
        ps_id = mc_rs.make_ps_id()
        ps = self.ps_storage.get(obj_id=ps_id, cluster=cluster)
        client = self.get_yp_client(cluster)
        if ps:
            validation.validate_pod_set_labels_matched(
                ps,
                self.match_labels,
                raise_if_not_matched=True
            )
        else:
            ps = podutil.make_pod_set(mc_rs=mc_rs,
                                      default_acl=self.default_acl,
                                      labels=self.match_labels,
                                      cluster=cluster)
            try:
                tid, _ = client.start_transaction()
                _, ps_fqid = client.create_pod_set(ps=ps, transaction_id=tid)
                self.mc_rs_client.add_deploy_child(mc_rs_id=mc_rs.meta.id, child=ps_fqid)
                client.commit_transaction(tid)
            except yp.common.YtResponseError:
                logger.info("failed to create podset in cluster %s, removing orphaned relation", cluster)
                self.mc_rs_client.remove_deploy_child(mc_rs_id=mc_rs.meta.id, child=ps_fqid)
                raise
            rel = yp.data_model.TRelation()
            # We do not use relation ID anywhere so generate some fake ID.
            rel.meta.id = randomutil.gen_random_str(bits=80)
            rel.meta.from_fqid = mc_rs.meta.fqid
            rel.meta.to_fqid = ps_fqid
            self.relation_storage.put(rel)
        client.create_pods(pods=pods, transaction_id=None)
        stat.created_pods += len(pods)
        logger.info('[%s] created %d pods', cluster, len(pods))

    def update_pods_in_cluster(self, mc_rs, pods, cluster, logger, stat):
        pods = pods[:self.get_max_pods_to_process(mc_rs)]
        logger.info('[%s] updating %d pods', cluster, len(pods))
        client = self.get_yp_client(cluster)
        client.update_pods(pods=pods, batch_size=self.update_pods_batch_size)
        stat.updated_pods += len(pods)
        logger.info('[%s] updated %d pods', cluster, len(pods))

    def reallocate_pods_in_cluster(self, mc_rs, pods, cluster, logger, stat):
        if len(pods) > self.get_max_pods_to_process(mc_rs):
            pods = pods[:self.get_max_pods_to_process(mc_rs)]

        logger.info('[%s] Trying to reallocate %d pods in-place', cluster, len(pods))
        client = self.get_yp_client(cluster)

        failed_pods = client.safe_update_pods(pods=pods)
        stat.reallocated_inplace_pods += len(pods) - len(failed_pods)

        if failed_pods:
            logger.info('[%s] In-place reallocation failed for %d pods, trying replacing', cluster, len(failed_pods))
            self.set_pods_target_state_removed_in_cluster(
                mc_rs=mc_rs,
                ids=[pod.meta.id for pod in failed_pods],
                cluster=cluster,
                logger=logger,
                stat=stat,
            )

    def replace_pods_in_cluster(self, mc_rs, ids, new_pods, cluster, logger, stat):
        ids = list(itertools.islice(ids, self.get_max_pods_to_process(mc_rs)))
        new_pods = new_pods[:self.get_max_pods_to_process(mc_rs)]
        logger.info('[%s] replacing %d pods', cluster, len(ids))
        client = self.get_yp_client(cluster)
        tid, ts = client.start_transaction()
        client.remove_pods(pod_ids=ids, transaction_id=tid)
        client.create_pods(pods=new_pods, transaction_id=tid)
        client.commit_transaction(tid)
        stat.replaced_pods += len(ids)
        logger.info('[%s] replaced %d pods', cluster, len(ids))

    def request_eviction_pods_in_cluster(self, mc_rs, ids, cluster, logger, stat):
        ids = list(itertools.islice(ids, self.get_max_pods_to_process(mc_rs)))
        logger.info('[%s] requesting eviction for %d pods', cluster, len(ids))
        client = self.get_yp_client(cluster)
        client.update_pods_request_eviction(pod_ids=ids, msg=self.EVICTION_REQUEST_MSG)
        stat.removed_pods += len(ids)
        logger.info('[%s] requested eviction for %d pods', cluster, len(ids))

    def mark_delegate_removing_pods_in_cluster(self, mc_rs, ids, cluster, logger, stat):
        ids = list(itertools.islice(ids, self.get_max_pods_to_process(mc_rs)))
        logger.info('[%s] delegate removing for %d pods', cluster, len(ids))
        client = self.get_yp_client(cluster)
        client.mark_delegate_removing_pods(pod_ids=ids)
        stat.removed_pods += len(ids)
        logger.info('[%s] delegate removing for %d pods', cluster, len(ids))

    def delegate_removing_pods_in_cluster(self, mc_rs, ids, cluster, logger, stat):
        ids = list(itertools.islice(ids, self.get_max_pods_to_process(mc_rs)))
        logger.info('[%s] delegate removing for %d pods', cluster, len(ids))
        client = self.get_yp_client(cluster)
        client.delegate_removing_pods(pod_ids=ids, msg=self.DELEGATE_REMOVING_MSG)
        stat.removed_pods += len(ids)
        logger.info('[%s] delegate removing for %d pods', cluster, len(ids))

    def acknowledge_eviction_pods_in_cluster(self, mc_rs, ids, cluster, logger, stat):
        ids = list(itertools.islice(ids, self.get_max_pods_to_process(mc_rs)))
        logger.info('[%s] acknowledging eviction of %d pods', cluster, len(ids))
        client = self.get_yp_client(cluster)
        evict_pod_ids, acknowledge_pod_ids = [], []

        for pod_id in ids:
            pod = self.pod_storage.get(pod_id, cluster)
            if pod.status.eviction.state == yp.data_model.ES_REQUESTED:
                acknowledge_pod_ids.append(pod_id)
            else:
                evict_pod_ids.append(pod_id)

        if evict_pod_ids:
            client.update_pods_acknowledge_eviction(
                pod_ids=evict_pod_ids,
                msg=self.ACKNOWLEDGE_MSG,
                use_evict=True
            )
            stat.eviction_acknowledged_pods += len(ids)
            logger.info('[%s] acknowledged eviction of %d pods', cluster, len(ids))

        if acknowledge_pod_ids:
            client.update_pods_acknowledge_eviction(
                pod_ids=acknowledge_pod_ids,
                msg=self.ACKNOWLEDGE_MSG,
                use_evict=False
            )
            stat.eviction_acknowledged_pods += len(ids)
            logger.info('[%s] acknowledged eviction of %d pods', cluster, len(ids))

    def acknowledge_maintenance_pods_in_cluster(self, mc_rs, ids, cluster, logger, stat):
        ids = list(itertools.islice(ids, self.get_max_pods_to_process(mc_rs)))
        logger.info('[%s] acknowledging maintenance of %d pods', cluster, len(ids))
        client = self.get_yp_client(cluster)
        client.update_pods_acknowledge_maintenance(
            pod_ids=ids,
            msg=self.ACKNOWLEDGE_MSG
        )
        stat.maintenance_acknowledged_pods += len(ids)
        logger.info('[%s] acknowledged maintenance of %d pods', cluster, len(ids))

    def set_pods_target_state_removed_in_cluster(self, mc_rs, ids, cluster, logger, stat):
        ids = list(itertools.islice(ids, self.get_max_pods_to_process(mc_rs)))
        logger.info('[%s] setting target state REMOVED %d pods', cluster, len(ids))
        client = self.get_yp_client(cluster)
        client.update_pods_target_state(pod_ids=ids, target_state=yp.data_model.EPodAgentTargetState_REMOVED)
        stat.set_target_state_removed_pods += len(ids)
        logger.info('[%s] set target state REMOVED %d pods', cluster, len(ids))

    def set_pods_target_state_active_in_cluster(self, mc_rs, ids, cluster, logger, stat):
        ids = list(itertools.islice(ids, self.get_max_pods_to_process(mc_rs)))
        logger.info('[%s] setting target state ACTIVE %d pods', cluster, len(ids))
        client = self.get_yp_client(cluster)
        client.update_pods_target_state(pod_ids=ids, target_state=yp.data_model.EPodAgentTargetState_ACTIVE)
        stat.set_target_state_active_pods += len(ids)
        logger.info('[%s] set target state ACTIVE %d pods', cluster, len(ids))

    def remove_pods_in_cluster(self, mc_rs, ids, cluster, logger, stat):
        ids = list(itertools.islice(ids, self.get_max_pods_to_process(mc_rs)))
        logger.info('[%s] removing %d pods', cluster, len(ids))
        client = self.get_yp_client(cluster)
        client.remove_pods(pod_ids=ids)
        stat.removed_pods += len(ids)
        logger.info('[%s] removed %d pods', cluster, len(ids))

    def update_pod_sets(self, mc_rs, ok_spec_clusters, logger, stat):
        tasks = []
        ps_id = mc_rs.make_ps_id()
        for cluster in ok_spec_clusters:
            ps = self.ps_storage.get(obj_id=ps_id, cluster=cluster)
            if not ps:
                continue

            is_matched = validation.validate_pod_set_labels_matched(
                ps,
                self.match_labels,
                raise_if_not_matched=False
            )
            if not is_matched:
                continue

            tpl = podutil.make_pod_set(mc_rs=mc_rs,
                                       default_acl=self.default_acl,
                                       labels=self.match_labels,
                                       cluster=cluster)
            if not podutil.update_pod_set_needed(ps, tpl):
                continue

            kw = {'ps_id': ps_id,
                  'template': tpl,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(task=self.update_pod_set_in_cluster,
                                        cluster=cluster,
                                        mc_rs_id=mc_rs.meta.id,
                                        check_rate_limiter=False,
                                        logger=logger,
                                        task_kwargs=kw)
            tasks.append(t)
        return tasks

    def remove_not_in_spec_pod_sets(self, mc_rs, not_in_spec_clusters,
                                    logger, stat):
        tasks = []
        ps_id = mc_rs.make_ps_id()
        for cluster in not_in_spec_clusters:
            ps = self.ps_storage.get(obj_id=ps_id, cluster=cluster)
            if not ps:
                continue
            is_matched = validation.validate_pod_set_labels_matched(
                ps,
                self.match_labels,
                raise_if_not_matched=False
            )
            if not is_matched:
                continue
            kw = {'ps': ps,
                  'mc_rs': mc_rs,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(task=self.remove_pod_set_in_cluster,
                                        cluster=cluster,
                                        mc_rs_id=mc_rs.meta.id,
                                        check_rate_limiter=False,
                                        logger=logger,
                                        task_kwargs=kw)
            tasks.append(t)
        return tasks

    def create_pods(self, action, mc_rs, logger, stat):
        tasks = []
        tpl = podutil.make_pod_template(mc_rs=mc_rs,
                                        match_labels=self.match_labels)
        for cluster, count in action.list_count_by_cluster():
            pods = podutil.make_pods(pod_template=tpl,
                                     count=count,
                                     pod_storage=self.pod_storage,
                                     cluster=cluster)
            kw = {'mc_rs': mc_rs,
                  'pods': pods,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(task=self.create_pods_in_cluster,
                                        cluster=cluster,
                                        mc_rs_id=mc_rs.meta.id,
                                        check_rate_limiter=True,
                                        logger=logger,
                                        task_kwargs=kw)
            tasks.append(t)
        return tasks

    def update_pods(self, action, mc_rs, logger, stat):
        tasks = []
        tpl = podutil.make_pod_template(mc_rs=mc_rs,
                                        match_labels=self.match_labels)
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            pods = podutil.make_pods_by_other_pods(pod_template=tpl,
                                                   pod_ids=ids,
                                                   pod_storage=self.pod_storage,
                                                   cluster=cluster)
            kw = {'mc_rs': mc_rs,
                  'pods': pods,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(task=self.update_pods_in_cluster,
                                        cluster=cluster,
                                        mc_rs_id=mc_rs.meta.id,
                                        check_rate_limiter=False,
                                        logger=logger,
                                        task_kwargs=kw)
            tasks.append(t)
        return tasks

    def update_pods_preserve_allocation(self, action, mc_rs, logger, stat):
        tasks = []
        tpl = podutil.make_pod_template(mc_rs=mc_rs,
                                        match_labels=self.match_labels)
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            pods = podutil.make_pods_by_other_pods_preserve_allocation(pod_template=tpl,
                                                                       pod_ids=ids,
                                                                       pod_storage=self.pod_storage,
                                                                       cluster=cluster)
            kw = {'mc_rs': mc_rs,
                  'pods': pods,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(task=self.update_pods_in_cluster,
                                        cluster=cluster,
                                        mc_rs_id=mc_rs.meta.id,
                                        check_rate_limiter=False,
                                        logger=logger,
                                        task_kwargs=kw)
            tasks.append(t)
        return tasks

    def replace_pods(self, action, mc_rs, logger, stat):
        tasks = []
        tpl = podutil.make_pod_template(mc_rs=mc_rs,
                                        match_labels=self.match_labels)
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            it = itertools.islice(ids, self.get_max_pods_to_process(mc_rs))
            for ids_batch in ctlutil.split_iterable_into_batches(it, self.replace_pods_batch_size):
                new_pods = podutil.make_pods_by_other_pods(pod_template=tpl,
                                                           pod_ids=ids_batch,
                                                           pod_storage=self.pod_storage,
                                                           cluster=cluster)
                kw = {'mc_rs': mc_rs,
                      'ids': ids_batch,
                      'new_pods': new_pods,
                      'cluster': cluster,
                      'logger': logger,
                      'stat': stat}
                t = self._make_cluster_task(task=self.replace_pods_in_cluster,
                                            cluster=cluster,
                                            mc_rs_id=mc_rs.meta.id,
                                            check_rate_limiter=True,
                                            logger=logger,
                                            task_kwargs=kw)
                tasks.append(t)
        return tasks

    def replace_pods_with_new_ids(self, action, mc_rs, logger, stat):
        tasks = []
        tpl = podutil.make_pod_template(mc_rs=mc_rs,
                                        match_labels=self.match_labels)
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            it = itertools.islice(ids, self.get_max_pods_to_process(mc_rs))
            for ids_batch in ctlutil.split_iterable_into_batches(it, self.replace_pods_batch_size):
                new_pods = podutil.make_pods(pod_template=tpl,
                                             count=len(ids_batch),
                                             pod_storage=self.pod_storage,
                                             cluster=cluster)
                kw = {'mc_rs': mc_rs,
                      'ids': ids_batch,
                      'new_pods': new_pods,
                      'cluster': cluster,
                      'logger': logger,
                      'stat': stat}
                t = self._make_cluster_task(task=self.replace_pods_in_cluster,
                                            cluster=cluster,
                                            mc_rs_id=mc_rs.meta.id,
                                            check_rate_limiter=True,
                                            logger=logger,
                                            task_kwargs=kw)
                tasks.append(t)
        return tasks

    def request_eviction_pods(self, action, mc_rs, logger, stat):
        tasks = []
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            kw = {'mc_rs': mc_rs,
                  'ids': ids,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(
                task=self.request_eviction_pods_in_cluster,
                cluster=cluster,
                mc_rs_id=mc_rs.meta.id,
                check_rate_limiter=False,
                logger=logger,
                task_kwargs=kw
            )
            tasks.append(t)
        return tasks

    def mark_delegate_removing_pods(self, action, mc_rs, logger, stat):
        tasks = []
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            kw = {'mc_rs': mc_rs,
                  'ids': ids,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(
                task=self.mark_delegate_removing_pods_in_cluster,
                cluster=cluster,
                mc_rs_id=mc_rs.meta.id,
                check_rate_limiter=False,
                logger=logger,
                task_kwargs=kw
            )
            tasks.append(t)
        return tasks

    def delegate_removing_pods(self, action, mc_rs, logger, stat):
        tasks = []
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            kw = {'mc_rs': mc_rs,
                  'ids': ids,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(
                task=self.delegate_removing_pods_in_cluster,
                cluster=cluster,
                mc_rs_id=mc_rs.meta.id,
                check_rate_limiter=False,
                logger=logger,
                task_kwargs=kw
            )
            tasks.append(t)
        return tasks

    def acknowledge_eviction_pods(self, action, mc_rs, logger, stat):
        tasks = []
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            kw = {'mc_rs': mc_rs,
                  'ids': ids,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(
                task=self.acknowledge_eviction_pods_in_cluster,
                cluster=cluster,
                mc_rs_id=mc_rs.meta.id,
                check_rate_limiter=False,
                logger=logger,
                task_kwargs=kw
            )
            tasks.append(t)
        return tasks

    def acknowledge_maintenance_pods(self, action, mc_rs, logger, stat):
        tasks = []
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            kw = {'mc_rs': mc_rs,
                  'ids': ids,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(
                task=self.acknowledge_maintenance_pods_in_cluster,
                cluster=cluster,
                mc_rs_id=mc_rs.meta.id,
                check_rate_limiter=False,
                logger=logger,
                task_kwargs=kw
            )
            tasks.append(t)
        return tasks

    def set_pods_target_state_removed(self, action, mc_rs, logger, stat):
        tasks = []
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            kw = {'mc_rs': mc_rs,
                  'ids': ids,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(task=self.set_pods_target_state_removed_in_cluster,
                                        cluster=cluster,
                                        mc_rs_id=mc_rs.meta.id,
                                        check_rate_limiter=True,
                                        logger=logger,
                                        task_kwargs=kw)
            tasks.append(t)
        return tasks

    def set_pods_target_state_active(self, action, mc_rs, logger, stat):
        tasks = []
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            kw = {'mc_rs': mc_rs,
                  'ids': ids,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(task=self.set_pods_target_state_active_in_cluster,
                                        cluster=cluster,
                                        mc_rs_id=mc_rs.meta.id,
                                        check_rate_limiter=False,
                                        logger=logger,
                                        task_kwargs=kw)
            tasks.append(t)
        return tasks

    def remove_pods(self, action, mc_rs, logger, stat):
        tasks = []
        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            kw = {'mc_rs': mc_rs,
                  'ids': ids,
                  'cluster': cluster,
                  'logger': logger,
                  'stat': stat}
            t = self._make_cluster_task(task=self.remove_pods_in_cluster,
                                        cluster=cluster,
                                        mc_rs_id=mc_rs.meta.id,
                                        check_rate_limiter=True,
                                        logger=logger,
                                        task_kwargs=kw)
            tasks.append(t)
        return tasks

    def reallocate_pods(self, action, mc_rs, logger, stat):
        tasks = []
        tpl = podutil.make_pod_template(mc_rs, match_labels=self.match_labels)

        for cluster, ids in action.list_grouped_by_cluster_pod_ids():
            it = itertools.islice(ids, self.get_max_pods_to_process(mc_rs))
            for ids_batch in ctlutil.split_iterable_into_batches(it, self.reallocate_pods_batch_size):
                pods = podutil.make_pods_by_other_pods(pod_template=tpl,
                                                       pod_ids=ids_batch,
                                                       pod_storage=self.pod_storage,
                                                       cluster=cluster)
                kw = {'mc_rs': mc_rs,
                      'pods': pods,
                      'cluster': cluster,
                      'logger': logger,
                      'stat': stat}
                t = self._make_cluster_task(task=self.reallocate_pods_in_cluster,
                                            cluster=cluster,
                                            mc_rs_id=mc_rs.meta.id,
                                            check_rate_limiter=False,
                                            logger=logger,
                                            task_kwargs=kw)
                tasks.append(t)

        return tasks

    def _make_tasks_by_pod_action(self, action, mc_rs, logger, stat):
        kind = action.kind
        if kind == PodActionKind.CreatePods:
            return self.create_pods(action=action,
                                    mc_rs=mc_rs,
                                    logger=logger,
                                    stat=stat)
        elif kind == PodActionKind.RemovePods:
            return self.remove_pods(action=action,
                                    mc_rs=mc_rs,
                                    logger=logger,
                                    stat=stat)
        elif kind == PodActionKind.RequestEvictionPods:
            return self.request_eviction_pods(action=action,
                                              mc_rs=mc_rs,
                                              logger=logger,
                                              stat=stat)
        elif kind == PodActionKind.MarkDelegateRemovingPods:
            return self.mark_delegate_removing_pods(action=action,
                                                    mc_rs=mc_rs,
                                                    logger=logger,
                                                    stat=stat)
        elif kind == PodActionKind.DelegateRemovingPods:
            return self.delegate_removing_pods(action=action,
                                               mc_rs=mc_rs,
                                               logger=logger,
                                               stat=stat)
        elif kind == PodActionKind.AcknowledgeEvictionPods:
            return self.acknowledge_eviction_pods(action=action,
                                                  mc_rs=mc_rs,
                                                  logger=logger,
                                                  stat=stat)
        elif kind == PodActionKind.AcknowledgeMaintenancePods:
            return self.acknowledge_maintenance_pods(action=action,
                                                     mc_rs=mc_rs,
                                                     logger=logger,
                                                     stat=stat)
        elif kind == PodActionKind.UpdatePods:
            return self.update_pods(action=action,
                                    mc_rs=mc_rs,
                                    logger=logger,
                                    stat=stat)
        elif kind == PodActionKind.UpdatePodsPreserveAllocation:
            return self.update_pods_preserve_allocation(action=action,
                                                        mc_rs=mc_rs,
                                                        logger=logger,
                                                        stat=stat)
        elif kind == PodActionKind.ReplacePods:
            return self.replace_pods(action=action,
                                     mc_rs=mc_rs,
                                     logger=logger,
                                     stat=stat)
        elif kind == PodActionKind.SetTargetStateRemovedPods:
            return self.set_pods_target_state_removed(action=action,
                                                      mc_rs=mc_rs,
                                                      logger=logger,
                                                      stat=stat)
        elif kind == PodActionKind.SetTargetStateActivePods:
            return self.set_pods_target_state_active(action=action,
                                                     mc_rs=mc_rs,
                                                     logger=logger,
                                                     stat=stat)
        elif kind == PodActionKind.ReplacePodsWithNewIds:
            return self.replace_pods_with_new_ids(action=action,
                                                  mc_rs=mc_rs,
                                                  logger=logger,
                                                  stat=stat)
        elif kind == PodActionKind.ReallocatePods:
            return self.reallocate_pods(action=action,
                                        mc_rs=mc_rs,
                                        logger=logger,
                                        stat=stat)
        else:
            raise exceptions.CtlError(
                'cannot apply pod task: unknown task kind: {}'.format(kind)
            )

    def unrelated_children_exist(self, mc_rs, logger, stat):
        for c in mc_rs.list_spec_clusters():
            ps = self.ps_storage.get(obj_id=mc_rs.meta.id, cluster=c)
            if not ps:
                continue
            rels = self.relation_storage.find(from_fqid=mc_rs.meta.fqid, to_fqid=ps.meta.fqid)
            if rels:
                continue
            stat.unrelated_children_found = True
            logger.error('mc_rs %s has no relation with pod_set %s in cluster %s',
                         mc_rs.meta.fqid, ps.meta.fqid, c)
            return True
        return False

    def is_update_allowed_by_min_deplay(self, mc_rs, log):
        speed = mc_rs.get_deploy_speed()
        if speed.min_delay == 0 and speed.update_portion == 0:
            return True
        cur_ts = time.time()
        ps_id = mc_rs.make_ps_id()
        for c in mc_rs.list_spec_clusters():
            ts = self.pod_storage.get_last_deploy_timestamp(ps_id, c)
            if not ts:
                continue
            delay = cur_ts - ts
            if delay > speed.min_delay:
                continue
            log.info('Not enough time has passed since last update: %s < %s', delay, speed.min_delay)
            return False
        return True

    def process(self, mc_rs, failed_clusters):
        clusters_info = self._make_clusters_info(mc_rs, failed_clusters)
        log = logging.getLogger('mc_rs({})'.format(mc_rs.meta.id))
        is_allowed = self.is_update_allowed_by_min_deplay(mc_rs, log)
        if not is_allowed:
            return

        log.info('clusters info: %s', clusters_info)

        current_state = self._make_current_state(
            mc_rs=mc_rs,
            ok_spec_clusters=clusters_info.ok_spec_clusters
        )
        log.info('current state: %s', current_state)
        log.info('starting process pods')

        template = mc_rs.spec.pod_template_spec
        target = mc_rs.spec_revision()
        # Copy mc_rs revision to pods.
        template.spec.pod_agent_payload.spec.revision = target

        tasks = []
        validation.validate_spec_clusters(mc_rs, self.clusters)

        stat = ProcessStatistic()
        has_unrelated_children = self.unrelated_children_exist(mc_rs, log, stat)
        if has_unrelated_children and self.stop_process_mc_rs_with_unrelated_children:
            log.error("unrelated children found, stop processing")
            return
        # Remove pods in clusters that are not in mc_rs spec.
        remove_ps_tasks = self.remove_not_in_spec_pod_sets(
            mc_rs,
            clusters_info.not_in_spec_clusters,
            logger=log,
            stat=stat
        )
        tasks.extend(remove_ps_tasks)

        # Update podsets (account_id, antiaffinity, acl) if they changed.
        update_ps_tasks = self.update_pod_sets(
            mc_rs=mc_rs,
            ok_spec_clusters=clusters_info.ok_spec_clusters,
            logger=log,
            stat=stat
        )
        tasks.extend(update_ps_tasks)

        unavailable_pods = 0
        for c in clusters_info.failed_spec_clusters:
            unavailable_pods += mc_rs.replica_count_by_cluster(c)

        deploy_policy = yputil.get_label(mc_rs.labels, consts.DEPLOY_POLICY_LABEL, 'default')
        current_policy = self.deploy_policies.get(deploy_policy, self.default_policy)
        actions = current_policy.process(
            current_state=current_state,
            mc_rs=mc_rs,
            pod_template=podutil.make_pod_template(mc_rs, self.match_labels),
            clusters=clusters_info.ok_spec_clusters,
            unavailable_pods=unavailable_pods,
            logger=log
        )
        for action in actions:
            tasks.extend(self._make_tasks_by_pod_action(action=action,
                                                        mc_rs=mc_rs,
                                                        logger=log,
                                                        stat=stat))
        ctl_errors = cluster_task.apply_cluster_tasks(tasks)
        status, is_ready = self.mc_rs_status_maker.make_status(
            mc_rs=mc_rs,
            current_state=current_state,
            ctl_errors=ctl_errors,
            failed_clusters=failed_clusters,
        )

        if status != mc_rs.status:
            log.info('updating mc_rs status: ready=%s, revision=%s', is_ready, target)
            try:
                self.mc_rs_client.update_status(mc_rs.meta.id, status)
                log.info('mc_rs status successfully updated')
            except Exception:
                log.exception('updating mc_rs status failed')
        else:
            log.info('mc_rs status has not changed: ready=%s, revision=%s', is_ready, target)
        stat.is_mc_rs_ready = is_ready
        return stat


def make_controller(deploy_engine, root_users,
                    yp_clients,
                    match_labels,
                    max_pods_to_process,
                    update_pods_batch_size,
                    replace_pods_batch_size,
                    reallocate_pods_batch_size,
                    clusters, mc_rs_cluster,
                    pod_storage, ps_storage, relation_storage,
                    circuit_breaker,
                    rate_limiter,
                    stop_process_mc_rs_with_unrelated_children=False):
    client = yp_clients[mc_rs_cluster]
    if deploy_engine == consts.RSC_DEPLOY_ENGINE:
        mc_rs_client = yp_client_adapter.RsYpClientAdapter(yp_client=client)
        mc_rs_status_maker = status_maker.ReplicaSetStatusMaker()
    elif deploy_engine == consts.MCRSC_DEPLOY_ENGINE:
        mc_rs_client = yp_client_adapter.McrsYpClientAdapter(yp_client=client)
        mc_rs_status_maker = status_maker.MultiClusterReplicaSetStatusMaker()
    else:
        raise ValueError('unknown deploy engine {}'.format(deploy_engine))
    return Controller(
        yp_clients=yp_clients,
        mc_rs_client=mc_rs_client,
        default_acl=[podutil.make_default_access_control_entry(root_users)],
        match_labels=match_labels,
        max_pods_to_process=max_pods_to_process,
        update_pods_batch_size=update_pods_batch_size,
        replace_pods_batch_size=replace_pods_batch_size,
        reallocate_pods_batch_size=reallocate_pods_batch_size,
        clusters=clusters,
        pod_storage=pod_storage,
        ps_storage=ps_storage,
        relation_storage=relation_storage,
        mc_rs_status_maker=mc_rs_status_maker,
        circuit_breaker=circuit_breaker,
        rate_limiter=rate_limiter,
        stop_process_mc_rs_with_unrelated_children=stop_process_mc_rs_with_unrelated_children
    )
