from __future__ import unicode_literals
import itertools

from infra.mc_rsc.src import consts
from infra.mc_rsc.src import podutil
from infra.mc_rsc.src import yputil
from infra.mc_rsc.src.pod_actions import (CreateAction,
                                          SetTargetStateActiveAction)


def unique_slice(iterable, count):
    if not count:
        return
    seen = set()
    for element in iterable:
        if element in seen:
            continue
        seen.add(element)
        count -= 1
        yield element
        if not count:
            break


class DeployPolicy(object):
    def __init__(self, pod_storage, pod_disruption_policy):
        self.pod_storage = pod_storage
        self.pod_disruption_policy = pod_disruption_policy

    @classmethod
    def calc_update_step(cls, current_state, mc_rs, unavailable_pods):
        target = mc_rs.spec_revision()
        return (mc_rs.max_unavailable() -
                current_state.in_progress.count(target) -
                current_state.failed.count(target) -
                unavailable_pods)

    @classmethod
    def max_tolerable_downtime_pods(cls, mc_rs):
        st = mc_rs.spec.deployment_strategy
        budget = st.max_tolerable_downtime_pods
        if budget < 1 or budget > mc_rs.max_unavailable():
            # We check budget <= mc_rs.max_unavailable() because there is no validation in YP
            budget = mc_rs.max_unavailable()
        return budget

    @classmethod
    def calc_evict_ready_step(cls, current_state, mc_rs, unavailable_pods):
        target = mc_rs.spec_revision()
        budget = cls.max_tolerable_downtime_pods(mc_rs)
        return (budget -
                current_state.in_progress.count(target) -
                current_state.failed.count(target) -
                unavailable_pods -
                current_state.implicitly_dead -
                current_state.target_state_removed.count_all())

    @classmethod
    def calc_evict_in_progress_step(cls, mc_rs, unavailable_pods):
        return cls.max_tolerable_downtime_pods(mc_rs) - unavailable_pods

    @classmethod
    def find_cluster_pod_ids_to_deploy(cls,
                                       current_state,
                                       target_revision,
                                       count):
        target_state_removed = current_state.target_state_removed
        in_progress = current_state.in_progress
        ready = current_state.ready
        failed = current_state.failed
        ids = itertools.chain(target_state_removed.exclude(target_revision),
                              current_state.to_evict_in_progress.exclude(target_revision),
                              current_state.to_evict_ready.exclude(target_revision),
                              failed.exclude(target_revision),
                              in_progress.exclude(target_revision),
                              ready.exclude(target_revision),
                              # We must add "current_state.removing_delegate_required" to pod ids
                              # to be processed and re-request eviction.
                              current_state.removing_delegate_required,
                              )
        return list(unique_slice(ids, count))

    @classmethod
    def find_cluster_pod_ids_to_activate(cls,
                                         current_state,
                                         target_revision,
                                         ):
        target_state_removed = current_state.target_state_removed
        ids = itertools.chain(target_state_removed.find(target_revision))
        return list(ids)

    @classmethod
    def find_cluster_pod_ids_to_remove(cls,
                                       current_state,
                                       target_revision,
                                       count):
        target_state_removed = current_state.target_state_removed
        in_progress = current_state.in_progress
        ready = current_state.ready
        failed = current_state.failed
        ids = itertools.chain(target_state_removed.all(),
                              current_state.to_evict_in_progress.all(),
                              current_state.to_evict_ready.all(),
                              failed.exclude(target_revision),
                              in_progress.exclude(target_revision),
                              ready.exclude(target_revision),
                              failed.find(target_revision),
                              in_progress.find(target_revision),
                              ready.find(target_revision))
        return list(unique_slice(ids, count))

    def _process_replica_count(self, current_state, mc_rs, clusters, is_set_removed_allowed, logger):
        create_action = CreateAction()
        actions = [create_action]
        target = mc_rs.spec_revision()
        for c in clusters:
            s = current_state.make_filtered_by_cluster_current_state(c)
            diff = s.count_all() - mc_rs.replica_count_by_cluster(c)
            if diff == 0:
                continue
            if diff < 0:
                create_action.add(cluster=c, count=abs(diff))
            else:
                to_remove = self.find_cluster_pod_ids_to_remove(
                    current_state=s,
                    target_revision=target,
                    count=diff
                )
                remove_actions = self.pod_disruption_policy.process_cluster_pod_ids_to_remove(
                    to_remove,
                    is_set_removed_allowed,
                    logger
                )
                actions.extend(remove_actions)

        return actions

    def _make_activate_pods_action(self, current_state, target_revision):
        activate_action = SetTargetStateActiveAction()
        to_activate_candidates = self.find_cluster_pod_ids_to_activate(current_state=current_state,
                                                                       target_revision=target_revision)
        for cluster, pod_id in to_activate_candidates:
            p = self.pod_storage.get(obj_id=pod_id, cluster=cluster)
            if not podutil.is_pod_eviction_empty(p):
                # Eviction is requested for pod, we must not start it again, let's wait a little
                # and acknowledge eviction or recreate pod if it won't be able to turn off
                continue
            if not podutil.is_maintenance_empty(p):
                # Maintenance is requested for pod, let's wait for this pod to be turned off
                # and not restart it
                continue
            if podutil.is_pod_node_alerted(p):
                # We have a problem with this node, let's better evict pod from it
                continue
            if podutil.is_pod_disabled_permanently(p):
                # This pod was frozen on MAN recovery, for details see: DEPLOY-5964
                continue
            activate_action.add(pod_id=pod_id, cluster=cluster)
        return activate_action

    def process(self, current_state, mc_rs, pod_template, clusters,
                unavailable_pods, logger):
        is_move_disabled = yputil.get_label(pod_template.labels, consts.DISABLE_PODS_MOVE_LABEL, False)
        is_set_removed_disabled = yputil.get_label(mc_rs.labels, consts.DISABLE_SET_TARGET_STATE_REMOVED_LABEL, False)
        logger.info("is_move_disabled: {}, is_set_removed_disabled: {}".format(
            is_move_disabled, is_set_removed_disabled))
        is_set_removed_allowed = not is_move_disabled and not is_set_removed_disabled

        # First of all we create new and remove redundand pods
        # and make real pods count equal to desired pods count given in spec
        actions = self._process_replica_count(
            current_state=current_state,
            mc_rs=mc_rs,
            clusters=clusters,
            is_set_removed_allowed=is_set_removed_allowed,
            logger=logger
        )
        create_remove = [a for a in actions if not a.is_empty()]
        if create_remove:
            return create_remove

        # After it we will process pods for which eviction or maintenance is requested
        # we must do it before new revision deployment because eviction requests have top priority
        evict_in_progress_step = self.calc_evict_in_progress_step(mc_rs=mc_rs, unavailable_pods=unavailable_pods)
        evict_ready_step = self.calc_evict_ready_step(current_state=current_state, mc_rs=mc_rs,
                                                      unavailable_pods=unavailable_pods)
        evict_actions = self.pod_disruption_policy.process_pod_ids_to_evict(
            current_state=current_state,
            evict_in_progress_step=evict_in_progress_step,
            evict_ready_step=evict_ready_step,
            is_set_removed_allowed=is_set_removed_allowed,
            logger=logger
        )
        if evict_actions:
            return evict_actions

        target_revision = mc_rs.spec_revision()

        # Now let's enable pods which were turned off for some reason:
        # 1. We turned them off and acknowledged eviction afterwards
        #    and now they must be activated again on new host
        # 2. Eviction request on these pods has been cancelled after we started to
        #    turn them off
        activate_action = self._make_activate_pods_action(current_state, target_revision)
        if not activate_action.is_empty():
            return [activate_action]

        # Now we must update pods onto new revision, here we do in-place updates
        # reallocation with graceful shutdown or pod replacement
        step = self.calc_update_step(current_state=current_state,
                                     mc_rs=mc_rs,
                                     unavailable_pods=unavailable_pods)
        # Count unavailable pods to limit pods to deploy:
        # https://st.yandex-team.ru/DEPLOY-2395.
        limit = mc_rs.replica_count() - current_state.count(target_revision) - unavailable_pods
        step = min(step, limit)
        if step <= 0:
            return []

        to_deploy = self.find_cluster_pod_ids_to_deploy(
            current_state=current_state,
            target_revision=target_revision,
            count=step
        )
        update_replace = self.pod_disruption_policy.process_cluster_pod_ids_to_deploy(
            to_deploy,
            pod_template,
            is_move_disabled,
            is_set_removed_allowed,
            logger
        )
        return update_replace
