
import itertools
from sepelib.core import config

from infra.mc_rsc.src import podutil
from infra.mc_rsc.src.pod_actions import (AcknowledgeEvictionAction,
                                          AcknowledgeMaintenanceAction,
                                          RemoveAction,
                                          ReplaceAction,
                                          ReplacePodsWithNewIdsAction,
                                          RequestEvictionAction,
                                          SetTargetStateRemovedAction,
                                          UpdateAction,
                                          ReallocateAction,
                                          UpdatePreserveAllocationAction,
                                          DelegateRemovingAction,
                                          MarkDelegateRemovingAction)


class IPodDisruptionPolicy(object):
    def process_cluster_pod_ids_to_remove(self, to_remove, is_set_removed_allowed, logger):
        """
        :type to_remove: list[tuple[str, str]]
        :type is_set_removed_allowed: bool
        :type logger: logging.Logger
        :rtype: list
        """
        raise NotImplementedError

    def process_cluster_pod_ids_to_deploy(self, to_deploy, pod_template, is_move_disabled,
                                          is_set_removed_allowed, logger):
        """
        :type to_deploy: list[tuple[str, str]
        :type pod_template: yp.data_model.TPod
        :type is_move_disabled: bool
        :type is_set_removed_allowed: bool
        :type logger: logging.Logger
        :rtype: list
        """
        raise NotImplementedError

    def process_pod_ids_to_evict(self, current_state, evict_in_progress_step, evict_ready_step,
                                 is_set_removed_allowed, logger):
        """
        :type current_state: infra.mc_rsc.src.state.MultiClusterCurrentState
        :type evict_in_progress_step: int
        :type evict_ready_step: int
        :type is_set_removed_allowed: bool
        :type logger: logging.Logger
        :rtype: list
        """
        raise NotImplementedError


class RollingUpdateDisruptionPolicy(IPodDisruptionPolicy):
    def __init__(self, pod_storage):
        self.pod_storage = pod_storage

    def process_cluster_pod_ids_to_remove(self, to_remove, is_set_removed_allowed, logger):
        remove_action = RemoveAction()
        set_removed_action = SetTargetStateRemovedAction()

        for cluster, pod_id in to_remove:
            p = self.pod_storage.get(obj_id=pod_id, cluster=cluster)
            if not is_set_removed_allowed or not podutil.is_pod_graceful_shutdown_required(p):
                remove_action.add(pod_id=pod_id, cluster=cluster)
                continue

            if podutil.is_target_state_removed(p):
                if podutil.is_removed_ready(p):
                    remove_action.add(pod_id=pod_id, cluster=cluster)
                elif podutil.is_destroy_overtimed(p):
                    logger.warning('destroy pod "%s" in "%s" overtimed and force removed',
                                   pod_id, cluster)
                    remove_action.add(pod_id=pod_id, cluster=cluster)
                else:
                    continue
            else:
                set_removed_action.add(pod_id=pod_id, cluster=cluster)

        return [remove_action, set_removed_action]

    def _process_disabled_pod(self, logger, cluster, pod_id, p, pod_template, update_action, replace_action, replace_overtimed_destroy):
        if podutil.is_pod_disabled_permanently(p):
            # Pod having "yd.disabled_permanently" label is a special case:
            # these pods have been turned off manually before MAN unfreezing: DEPLOY-5945.
            # In other cases we would just recreate these pods and reschedule them on new host
            # but in this special case we would reschedule all the pods in cluster (~100k) and
            # it's possible that scheduler won't be able to solve this problem.
            # So we should try to update them without rescheduling to save some scheduler work
            if podutil.is_pod_spec_updateable(p.spec, pod_template.spec):
                update_action.add(pod_id=pod_id, cluster=cluster)
                return
            replace_action.add(pod_id=pod_id, cluster=cluster)
            return
        if podutil.is_removed_ready(p):
            # Pod has target_state = REMOVED so it looks like we started turning it off
            # on previous iterations to recreate it. We see that it has condition ready == true,
            # so it's turned off already and now we can just recreate it
            replace_action.add(pod_id=pod_id, cluster=cluster)
            return
        if podutil.is_destroy_overtimed(p):
            logger.warning('pod "%s" in "%s" destroying is overtimed, will be forcefully replaced', pod_id, cluster)
            replace_overtimed_destroy.add(pod_id=pod_id, cluster=cluster)
            return

    def process_cluster_pod_ids_to_deploy(self, to_deploy, pod_template, is_move_disabled,
                                          is_set_removed_allowed, logger):
        update_action = UpdateAction()
        reallocate_action = ReallocateAction()
        replace_action = ReplaceAction()
        replace_overtimed_destroy = ReplacePodsWithNewIdsAction()
        set_removed_action = SetTargetStateRemovedAction()

        for cluster, pod_id in to_deploy:
            p = self.pod_storage.get(obj_id=pod_id, cluster=cluster)
            if podutil.is_target_state_removed(p):
                self._process_disabled_pod(logger, cluster, pod_id, p, pod_template, update_action, replace_action, replace_overtimed_destroy)
            elif is_move_disabled or podutil.is_pod_spec_updateable(p.spec, pod_template.spec):
                # Important: pod must not be in removed state
                update_action.add(pod_id=pod_id, cluster=cluster)
            elif podutil.is_pod_spec_updateable(p.spec, pod_template.spec, allow_resources=True):
                reallocate_action.add(pod_id=pod_id, cluster=cluster)
            elif is_set_removed_allowed and podutil.is_pod_graceful_shutdown_required(p):
                set_removed_action.add(pod_id=pod_id, cluster=cluster)
            else:
                replace_action.add(pod_id=pod_id, cluster=cluster)

        update_replace = []
        if not replace_overtimed_destroy.is_empty():
            update_replace.append(replace_overtimed_destroy)
        elif not update_action.is_empty():
            update_replace.append(update_action)
        elif not set_removed_action.is_empty():
            update_replace.append(set_removed_action)
        elif not reallocate_action.is_empty():
            update_replace.append(reallocate_action)
        elif not replace_action.is_empty():
            update_replace.append(replace_action)

        return update_replace

    def process_pod_ids_to_evict(self, current_state, evict_in_progress_step, evict_ready_step,
                                 is_set_removed_allowed, logger):
        is_eviction_disabled = config.get_value('controller.disable_eviction', False)
        threshold = config.get_value('controller.soft_maintenance_pods_threshold', 2)
        is_maintenance_disabled = current_state.count_all() < threshold
        if not is_eviction_disabled and evict_in_progress_step > 0:
            evict_action = AcknowledgeEvictionAction()
            set_removed_action = SetTargetStateRemovedAction()
            acknowledge_maintenance_action = AcknowledgeMaintenanceAction()

            if current_state.target_state_removed.count_all():
                for cluster, pod_id in current_state.target_state_removed.all():
                    p = self.pod_storage.get(obj_id=pod_id, cluster=cluster)
                    has_eviction_or_maintenance = (
                        podutil.is_pod_node_alerted(p) or
                        podutil.is_maintenance_overtimed(p, current_state.max_tolerable_downtime_seconds) or
                        podutil.is_maintenance_in_progress(p) or
                        podutil.is_pod_eviction_requested(p) or
                        podutil.is_maintenance_requested(p))
                    if not has_eviction_or_maintenance or podutil.is_pod_eviction_acknowledged(p):
                        continue

                    if podutil.is_removed_ready(p):
                        evict_action.add(cluster=cluster, pod_id=pod_id)
                    elif podutil.is_destroy_overtimed(p):
                        evict_action.add(cluster=cluster, pod_id=pod_id)
                        logger.warning('destroy pod "%s" in "%s" is overtimed and force evicted', pod_id, cluster)
                    else:
                        continue

                if not evict_action.is_empty():
                    return [evict_action]

            if current_state.node_alerted.count_all():
                # don't take budget into account, pod is dead already
                cluster, pod_id = current_state.node_alerted.first()
                return self.process_pod_eviction(
                    cluster=cluster,
                    pod_id=pod_id,
                    set_removed_action=set_removed_action,
                    evict_action=evict_action,
                    is_set_removed_allowed=is_set_removed_allowed
                )
            if current_state.maintenance_overtimed.count_all():
                # don't take budget into account, pod is dead already
                cluster, pod_id = current_state.maintenance_overtimed.first()
                return self.process_pod_eviction(
                    cluster=cluster,
                    pod_id=pod_id,
                    set_removed_action=set_removed_action,
                    evict_action=evict_action,
                    is_set_removed_allowed=is_set_removed_allowed
                )

            maintenances = sorted(current_state.maintenances.itervalues(), reverse=True)
            for m in maintenances:
                if not (m.waiting_in_progress.count_all() or m.waiting_ready.count_all()):
                    # Nothing to process
                    continue
                # Each maintenance affects N pods. We have two different cases:
                # 1. N <= disruption budget. We can acknowledge such maintenance for all affected pods.
                # 2. N > disruption budget. We have to evict (N - disruption_budget) pods and acknowledge
                #    maintenance for remaining.
                if m.waiting_in_progress.count_all() <= evict_in_progress_step and (m.waiting_ready.count_all() <=
                                                                                    evict_ready_step):
                    # Here we acknowledge maintenances ONLY of type (2)
                    if m.waiting_in_progress.count_all():
                        cluster, pod_id = m.waiting_in_progress.first()
                        if m.disruptive or is_maintenance_disabled:
                            return self.process_pod_eviction(
                                cluster=cluster,
                                pod_id=pod_id,
                                set_removed_action=set_removed_action,
                                evict_action=evict_action,
                                is_set_removed_allowed=is_set_removed_allowed
                            )
                        else:
                            acknowledge_maintenance_action.add(cluster=cluster, pod_id=pod_id)
                            return [acknowledge_maintenance_action]
                    if m.waiting_ready.count_all():
                        cluster, pod_id = m.waiting_ready.first()
                        if m.disruptive or is_maintenance_disabled:
                            return self.process_pod_eviction(
                                cluster=cluster,
                                pod_id=pod_id,
                                set_removed_action=set_removed_action,
                                evict_action=evict_action,
                                is_set_removed_allowed=is_set_removed_allowed
                            )
                        else:
                            acknowledge_maintenance_action.add(cluster=cluster, pod_id=pod_id)
                            return [acknowledge_maintenance_action]
                if m.acknowledged.count_all():
                    # Here we evict pods for maintenances of type (1)
                    # And we don't take budget into account, pod may become dead at any moment
                    cluster, pod_id = m.acknowledged.first()
                    return self.process_pod_eviction(
                        cluster=cluster,
                        pod_id=pod_id,
                        set_removed_action=set_removed_action,
                        evict_action=evict_action,
                        is_set_removed_allowed=is_set_removed_allowed
                    )
                if m.waiting_in_progress.count_all():
                    # don't take budget into account, pod is dead already
                    cluster, pod_id = m.waiting_in_progress.first()
                    return self.process_pod_eviction(
                        cluster=cluster,
                        pod_id=pod_id,
                        set_removed_action=set_removed_action,
                        evict_action=evict_action,
                        is_set_removed_allowed=is_set_removed_allowed
                    )

            # Now we have only maintenances for which all the following conditions are correct:
            # 1. m.acknowledged == []
            # 2. m.waiting_dead == []
            # 3. len(m.waiting_alive) > max_pods_to_process
            if evict_ready_step > 0:
                for m in maintenances:
                    if m.waiting_ready.count_all():
                        cluster, pod_id = m.waiting_ready.first()
                        return self.process_pod_eviction(
                            cluster=cluster,
                            pod_id=pod_id,
                            set_removed_action=set_removed_action,
                            evict_action=evict_action,
                            is_set_removed_allowed=is_set_removed_allowed
                        )

            for cluster, pod_id in itertools.islice(current_state.to_evict_in_progress.all(), evict_in_progress_step):
                self.process_pod_eviction(
                    cluster=cluster,
                    pod_id=pod_id,
                    set_removed_action=set_removed_action,
                    evict_action=evict_action,
                    is_set_removed_allowed=is_set_removed_allowed
                )
            if evict_ready_step > 0:
                # take budget into account
                for cluster, pod_id in itertools.islice(current_state.to_evict_ready.all(), evict_ready_step):
                    self.process_pod_eviction(
                        cluster=cluster,
                        pod_id=pod_id,
                        set_removed_action=set_removed_action,
                        evict_action=evict_action,
                        is_set_removed_allowed=is_set_removed_allowed
                    )
            if not evict_action.is_empty():
                return [evict_action]
            if not set_removed_action.is_empty():
                return [set_removed_action]

        # nothing to do here
        return []

    def process_pod_eviction(self, cluster, pod_id, set_removed_action, evict_action, is_set_removed_allowed):
        pod = self.pod_storage.get(obj_id=pod_id, cluster=cluster)
        if is_set_removed_allowed and podutil.is_pod_graceful_shutdown_required(pod):
            set_removed_action.add(cluster=cluster, pod_id=pod_id)
            return [set_removed_action]
        else:
            evict_action.add(cluster=cluster, pod_id=pod_id)
            return [evict_action]


class RemoveDelegationDisruptionPolicy(IPodDisruptionPolicy):
    def __init__(self, pod_storage, parallel_reallocation=False):
        self.pod_storage = pod_storage
        self.parallel_reallocation = parallel_reallocation

    def process_cluster_pod_ids_to_remove(self, to_remove, is_set_removed_allowed, logger):
        set_eviction_action = RequestEvictionAction()

        for cluster, pod_id in to_remove:
            p = self.pod_storage.get(obj_id=pod_id, cluster=cluster)
            if podutil.is_pod_eviction_requested(p):
                continue
            set_eviction_action.add(pod_id=pod_id, cluster=cluster)

        return [set_eviction_action]

    def process_cluster_pod_ids_to_deploy(self, to_deploy, pod_template, is_move_disabled,
                                          is_set_removed_allowed, logger):
        if self.parallel_reallocation:
            return self._process_cluster_pod_ids_to_deploy_parallel_reallocation(
                to_deploy, pod_template, is_move_disabled, is_set_removed_allowed, logger)

        return self._process_cluster_pod_ids_to_deploy(
            to_deploy, pod_template, is_move_disabled, is_set_removed_allowed, logger)

    def _process_cluster_pod_ids_to_deploy(self, to_deploy, pod_template, is_move_disabled,
                                           is_set_removed_allowed, logger):
        update_action = UpdateAction()
        request_eviction_action = RequestEvictionAction()

        for cluster, pod_id in to_deploy:
            p = self.pod_storage.get(obj_id=pod_id, cluster=cluster)
            if podutil.is_pod_spec_updateable(p.spec, pod_template.spec):
                # Processing in-place update even if eviction was requested
                # https://st.yandex-team.ru/DEPLOY-3726#5fdc9fc35f4d777564df55ff
                update_action.add(pod_id=pod_id, cluster=cluster)
            elif not podutil.is_pod_eviction_requested(p):
                request_eviction_action.add(pod_id=pod_id, cluster=cluster)

        update_replace = []
        if not update_action.is_empty():
            update_replace.append(update_action)
        elif not request_eviction_action.is_empty():
            update_replace.append(request_eviction_action)

        return update_replace

    def _process_cluster_pod_ids_to_deploy_parallel_reallocation(self, to_deploy, pod_template, is_move_disabled,
                                                                 is_set_removed_allowed, logger):
        """
        In "parallel reallocation" mode RSC must do the following:
        1. Update all pod spec fields *except* allocation (CPU, memory, disk, etc)
        2. If new RS allocation is not equal to pod allocation, it *must not* update allocation and must request eviction.
        """
        update_preserve_allocation_action = UpdatePreserveAllocationAction()
        delegate_removing_action = DelegateRemovingAction()
        mark_delegate_removing_action = MarkDelegateRemovingAction()
        for cluster, pod_id in to_deploy:
            p = self.pod_storage.get(obj_id=pod_id, cluster=cluster)

            if podutil.is_pod_spec_updateable(p.spec, pod_template.spec):
                # Case 1: RS allocation == pod allocation => just update spec
                update_preserve_allocation_action.add(pod_id=pod_id, cluster=cluster)
                continue
            if not podutil.is_pod_eviction_requested_ignore_reason(p):
                # Case 2: RS allocation != pod allocation =>
                # 1. Request eviction and do not update revision
                # 2. We will update spec on the next iteration
                delegate_removing_action.add(pod_id=pod_id, cluster=cluster)
                continue
            if not podutil.is_pod_marked_removing_delegate(p):
                # Case 3: RS allocation != pod allocation and eviction has been revoked => re-request eviction
                mark_delegate_removing_action.add(pod_id=pod_id, cluster=cluster)
                continue
            # Case 4: RS allocation != pod allocation and eviction already requested =>
            # update spec and revision preserving allocation
            update_preserve_allocation_action.add(pod_id=pod_id, cluster=cluster)

        update_replace = []
        if not update_preserve_allocation_action.is_empty():
            update_replace.append(update_preserve_allocation_action)
        elif not delegate_removing_action.is_empty():
            update_replace.append(delegate_removing_action)
        elif not mark_delegate_removing_action.is_empty():
            update_replace.append(mark_delegate_removing_action)

        return update_replace

    def process_pod_ids_to_evict(self, current_state, evict_in_progress_step, evict_ready_step,
                                 is_set_removed_allowed, logger):
        # TODO: Soft maintenance is not supported for this policy yet (at least for its 1st version).
        # TODO: We will support soft maintenance for this policy if base search asks to do it, but for now
        # TODO: we have agreement to start without it https://st.yandex-team.ru/DEPLOY-3726
        return []
