from __future__ import unicode_literals
import logging

import gevent
import yt.common
from infra.swatlib.gevent import greenthread
from infra.swatlib.gevent import geventutil as gutil
from infra.mc_rsc.src import consts


class PodSetGc(greenthread.GreenThread):

    MC_RS_SELECTORS = ['/meta/id']

    def __init__(self, name, cluster, client, mc_rs_getter,
                 mc_rs_storage, ps_storage, sleep_secs,
                 metrics_registry):

        super(PodSetGc, self).__init__()
        self.name = name
        self.cluster = cluster
        self.client = client
        self.mc_rs_getter = mc_rs_getter
        self.mc_rs_storage = mc_rs_storage
        self.ps_storage = ps_storage
        self.sleep_secs = sleep_secs
        self.log = logging.getLogger('{}({})'.format(self.name, cluster))
        self._metrics_registry = metrics_registry

    def check_pod_set(self, ps_id):
        # TODO: make mc_rs_id from ps_id
        mc_rs_id = ps_id
        try:
            mc_rs = self.mc_rs_getter(mc_rs_id,
                                      selectors=self.MC_RS_SELECTORS)
        except Exception:
            return False
        return mc_rs is None

    def remove_pod_sets(self, ids):
        # Do not check pod_sets' labels here because these pod_sets
        # were selected with proper match_filter.
        removed = set()
        metric = 'removed_gc_pod_sets-{}'.format(self.cluster)
        counter = self._metrics_registry.get_counter(metric)
        for ps_id in ids:
            try:
                if not self.check_pod_set(ps_id):
                    continue
                self.client.remove_pod_set(ps_id)
                counter.inc(1)
                removed.add(ps_id)
            except (yt.common.YtError, yt.common.YtResponseError) as e:
                self.log.exception("removing podset '%s' "
                                   "failed with error: %s", ps_id, e)
        return removed

    def process_gc(self):
        existing_ps_ids = set()
        for mc_rs in gutil.gevent_idle_iter(self.mc_rs_storage.list()):
            # TODO: make ps_id from mc_rs_id
            existing_ps_ids.add(mc_rs.meta.id)
        remove_candidates = set()
        for ps in gutil.gevent_idle_iter(self.ps_storage.list()):
            ps_id = ps.meta.id
            if ps_id not in existing_ps_ids:
                remove_candidates.add(ps_id)
        return self.remove_pod_sets(remove_candidates)

    def run(self):
        while True:
            try:
                removed = self.process_gc()
            except Exception as e:
                self.log.exception("gc failed with error: %s", e)
            else:
                self.log.info("removed podsets: %s", ",".join(removed))
            gevent.sleep(self.sleep_secs)


def make_ps_gc(deploy_engine,
               name, xdc_client, cluster, client,
               mc_rs_storage, ps_storage, sleep_secs,
               metrics_registry):
    if deploy_engine == consts.RSC_DEPLOY_ENGINE:
        mc_rs_getter = xdc_client.get_replica_set_ignore
    elif deploy_engine == consts.MCRSC_DEPLOY_ENGINE:
        mc_rs_getter = xdc_client.get_multi_cluster_replica_set_ignore
    else:
        raise ValueError('unknown deploy engine {}'.format(deploy_engine))
    return PodSetGc(name=name,
                    cluster=cluster,
                    client=client,
                    mc_rs_getter=mc_rs_getter,
                    mc_rs_storage=mc_rs_storage,
                    ps_storage=ps_storage,
                    sleep_secs=sleep_secs,
                    metrics_registry=metrics_registry)
