from google.protobuf import message as pb_message
from google.protobuf import pyext as pb_pyext

import yp.client
import yt.yson as yson
import yt_yson_bindings
from yp_proto.yp.client.api.proto import object_service_pb2

from infra.mc_rsc.src import consts


def make_attr_paths(selectors):
    return tuple(s.strip("/").split("/") for s in selectors)


def make_loader(payload_format):
    if payload_format == 'yson':
        return YsonLoader()
    if payload_format == 'protobuf':
        return PbLoader()
    raise ValueError('payload must be "yson" or "protobuf", got "{}"'.format(payload_format))


class YsonLoader(object):
    _enum_values_index = {}
    ID_SELECTORS = ['/meta/id']
    ID_ATTR_PATHS = make_attr_paths(ID_SELECTORS)
    PAYLOAD_FORMAT = object_service_pb2.PF_YSON

    @staticmethod
    def get_selectors_by_object_type(data_type):
        if data_type == yp.data_model.OT_POD:
            return consts.POD_SELECTORS
        return consts.DEFAULT_OBJECT_SELECTORS

    @staticmethod
    def get_updated_time_selector_by_object_type(data_type):
        if data_type == yp.data_model.OT_POD:
            return consts.POD_AGENT_SPEC_REVISION_PATH
        return '/spec'

    @classmethod
    def load_object(cls, obj_class, selectors, value_payloads):
        paths = make_attr_paths(selectors)
        return cls.load_object_attrs(obj_class, paths, value_payloads)

    @classmethod
    def load_object_attrs(cls, obj_class, paths, value_payloads):
        obj = obj_class()
        for i, path in enumerate(paths):
            cls._set_object_attr(obj, path, value_payloads[i].yson)
        return obj

    @classmethod
    def _set_object_attr(cls, obj, attr_path, yson_value):
        parent = None
        attr_name = None
        attr = obj
        for attr_name in attr_path:
            if attr_name:
                parent = attr
                attr = getattr(attr, attr_name)

        if isinstance(attr, pb_message.Message):
            yt_yson_bindings.loads_proto(yson_value, proto_object=attr, skip_unknown_fields=True)
            return

        value = yson.loads(yson_value)
        if isinstance(value, yson.yson_types.YsonEntity):
            return

        if isinstance(attr, pb_pyext._message.RepeatedCompositeContainer):
            for d in value:
                item = attr.add()
                dumped = yson.dumps(d)
                yt_yson_bindings.loads_proto(dumped, proto_object=item, skip_unknown_fields=True)
        elif isinstance(attr, pb_pyext._message.RepeatedScalarContainer):
            attr.extend(value)
        elif attr_name and parent.DESCRIPTOR.fields_by_name[attr_name].enum_type:
            enum_type = parent.DESCRIPTOR.fields_by_name[attr_name].enum_type
            enum_full_name = enum_type.full_name
            if enum_full_name not in cls._enum_values_index:
                cls._update_enum_values_index(enum_type)
            enum_values = cls._enum_values_index[enum_full_name]
            if value not in enum_values:
                raise yp.client.YpClientError('Value "{}" not defined for enum "{}"'.format(value, enum_full_name))
            setattr(parent, attr_name, enum_values[value])
        else:
            setattr(parent, attr_name, value)

    @classmethod
    def _update_enum_values_index(cls, enum_type):
        enum_values = {}
        for enum_value in enum_type.values:
            ext_value = yp.client.get_proto_enum_value_name(enum_value)
            enum_values[ext_value] = enum_value.number
        cls._enum_values_index[enum_type.full_name] = enum_values


class PbLoader(object):
    ID_SELECTORS = ['/meta']
    ID_ATTR_PATHS = make_attr_paths(ID_SELECTORS)
    PAYLOAD_FORMAT = object_service_pb2.PF_PROTOBUF

    @staticmethod
    def get_selectors_by_object_type(data_type):
        return consts.DEFAULT_OBJECT_SELECTORS

    @staticmethod
    def get_updated_time_selector_by_object_type(data_type):
        return '/spec'

    @classmethod
    def load_object(cls, obj_class, selectors, value_payloads):
        paths = make_attr_paths(selectors)
        return cls.load_object_attrs(obj_class, paths, value_payloads)

    @classmethod
    def load_object_attrs(cls, obj_class, paths, value_payloads):
        obj = obj_class()
        for i, path in enumerate(paths):
            cls._set_object_attr(obj, path, value_payloads[i].protobuf)
        return obj

    @staticmethod
    def _set_object_attr(obj, attr_path, protobuf_value):
        attr = obj
        for attr_name in attr_path:
            if attr_name:
                attr = getattr(attr, attr_name)
        attr.ParseFromString(protobuf_value)
