from __future__ import unicode_literals
import collections
import math

from infra.mc_rsc.src import yputil


DeploySpeed = collections.namedtuple('DeploySpeed', ['update_portion', 'min_delay'])

EMPTY_DEPLOY_SPEED = DeploySpeed(0, 0)


def get_deploy_speed_from_labels(labels):
    d = yputil.get_label(labels, 'yd.deploy_speed')
    if not d:
        return EMPTY_DEPLOY_SPEED
    update_portion = int(d.get('update_portion', 0))
    if update_portion < 0:
        update_portion = 0
    min_delay = int(d.get('min_delay', 0))
    if min_delay < 0:
        min_delay = 0
    if update_portion == 0 and min_delay == 0:
        return EMPTY_DEPLOY_SPEED
    return DeploySpeed(update_portion=update_portion, min_delay=min_delay)


def _make_disabled_clusters_from_labels(labels):
    v = yputil.get_label(labels, 'deploy') or {}
    return v.get('disabled_clusters') or set()


class ReplicaSet(object):

    def __init__(self, obj, cluster):
        self.rs = obj
        self.cluster = cluster

    @property
    def meta(self):
        return self.rs.meta

    @property
    def spec(self):
        return self.rs.spec

    @property
    def status(self):
        return self.rs.status

    @property
    def labels(self):
        return self.rs.labels

    @property
    def annotations(self):
        return self.rs.annotations

    def make_ps_id(self):
        return self.meta.id

    def spec_revision(self):
        return int(self.rs.spec.revision_id)

    def has_cluster(self, cluster):
        return cluster == self.cluster

    def list_spec_clusters(self):
        return [self.cluster]

    def replica_count(self):
        return self.rs.spec.replica_count

    def replica_count_by_cluster(self, cluster):
        return self.rs.spec.replica_count

    def constraints(self, cluster):
        return self.rs.spec.constraints

    def max_unavailable(self):
        return self.rs.spec.deployment_strategy.max_unavailable

    def get_deploy_speed(self):
        if self.rs.spec.deployment_strategy.HasField('deploy_speed'):
            return self.rs.spec.deployment_strategy.deploy_speed
        return get_deploy_speed_from_labels(self.rs.labels)

    def disabled_clusters(self):
        if hasattr(self, '_disabled_clusters'):
            return self._disabled_clusters
        self._disabled_clusters = _make_disabled_clusters_from_labels(self.rs.labels)
        return self._disabled_clusters

    def __str__(self):
        return str(self.rs)


class MultiClusterReplicaSet(object):

    def __init__(self, obj, cluster):
        self.mcrs = obj
        self.cluster = cluster
        disabled_clusters = self.disabled_clusters()
        self._specs = {}
        for c in self.mcrs.spec.clusters:
            if c.cluster in disabled_clusters:
                continue
            self._specs[c.cluster] = c.spec

    @property
    def meta(self):
        return self.mcrs.meta

    @property
    def spec(self):
        return self.mcrs.spec

    @property
    def status(self):
        return self.mcrs.status

    @property
    def labels(self):
        return self.mcrs.labels

    @property
    def annotations(self):
        return self.mcrs.annotations

    def make_ps_id(self):
        return self.meta.id

    def spec_revision(self):
        return self.mcrs.spec.revision

    def has_cluster(self, cluster):
        return cluster in self._specs

    def list_spec_clusters(self):
        return self._specs.keys()

    def replica_count(self):
        return sum(s.replica_count for s in self._specs.itervalues())

    def disabled_clusters(self):
        if hasattr(self, '_disabled_clusters'):
            return self._disabled_clusters
        self._disabled_clusters = _make_disabled_clusters_from_labels(self.mcrs.labels)
        return self._disabled_clusters

    def replica_count_by_cluster(self, cluster):
        return self._specs[cluster].replica_count

    def constraints(self, cluster):
        return self._specs[cluster].constraints

    def max_unavailable(self):
        if hasattr(self, '_max_unavailble'):
            return self._max_unavailable
        max_unavailable = self.mcrs.spec.deployment_strategy.max_unavailable
        # https://st.yandex-team.ru/DEPLOY-5820. Without this later max function will set 1 to _max_unavailable
        if max_unavailable == 0:
            self._max_unavailable = max_unavailable
            return self._max_unavailable
        if not self.disabled_clusters():
            self._max_unavailable = max_unavailable
            return self._max_unavailable
        total_pods = 0
        disabled_pods = 0
        for c in self.mcrs.spec.clusters:
            total_pods += c.spec.replica_count
            if c.cluster in self.disabled_clusters():
                disabled_pods += c.spec.replica_count
        # https://st.yandex-team.ru/DEPLOY-5825. Avoid division by zero.
        if total_pods == 0:
            return max_unavailable
        self._max_unavailable = max(math.floor(max_unavailable * (total_pods - disabled_pods) / float(total_pods)), 1)
        return self._max_unavailable

    def get_deploy_speed(self):
        if self.mcrs.spec.deployment_strategy.HasField('deploy_speed'):
            return self.mcrs.spec.deployment_strategy.deploy_speed
        return get_deploy_speed_from_labels(self.mcrs.labels)

    def __str__(self):
        return str(self.mcrs)
