from __future__ import unicode_literals
import collections
import hashlib
import itertools
import string

from infra.swatlib import randomutil
from infra.swatlib.gevent import geventutil as gutil


class IPodIdGenerator(object):

    def generate(self):
        raise NotImplementedError


class RandomPodIdGenerator(IPodIdGenerator):

    def generate(self):
        return randomutil.gen_random_str(bits=80)


RANDOM_POD_ID_GENERATOR = RandomPodIdGenerator()


EnumeratedPodId = collections.namedtuple('EnumeratedPodId', ['id', 'index'])


def generate_string_by_other_string(s, length, alphabet):
    total = len(alphabet)
    num = int(hashlib.sha1(s).hexdigest(), 16)
    chars = []
    for _ in xrange(length):
        r, num = num % total, num / total
        chars.append(alphabet[r])
    return ''.join(chars)


class EnumeratedPodIdGenerator(IPodIdGenerator):

    # TODO: move these settings to config.
    ID_PREFIX_MAX_LENGTH = 32
    DEFAULT_SALT_LENGTH = 8
    DEFAULT_SALT_ALPHABET = string.lowercase + string.digits

    def __init__(self, pod_set_id, pod_storage, cluster):
        self.prefix = self._make_salted_prefix_from_pod_set_id(pod_set_id)
        self.pod_storage = pod_storage
        self.cluster = cluster
        self._current_index = 0

    def _make_salted_prefix_from_pod_set_id(self, pod_set_id):
        salt = generate_string_by_other_string(pod_set_id,
                                               self.DEFAULT_SALT_LENGTH,
                                               self.DEFAULT_SALT_ALPHABET)
        norm_ps_id = pod_set_id.lower().replace('.', '-')
        prefix = norm_ps_id[:self.ID_PREFIX_MAX_LENGTH - self.DEFAULT_SALT_LENGTH - 1].rstrip('-')
        return '{}-{}'.format(prefix, salt)

    def generate(self):
        for idx in gutil.gevent_idle_iter(itertools.count(self._current_index)):
            p_id = '{}-{}'.format(self.prefix, idx)
            if self.pod_storage.get(p_id, self.cluster):
                continue
            self._current_index = idx + 1
            return EnumeratedPodId(p_id, idx)
